<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Risk scorer (supports multiple scoring strategies and unit-test injection).
 *
 * @package     local_behavioranalytics
 * @category    analytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\local\risk;

use local_behavioranalytics\local\detector\registry;
use stdClass;

/**
 * Computes user risk scores from anomaly detector findings.
 *
 * This class aggregates the results of registered detectors into a unified
 * numerical risk score and qualitative level. It supports multiple scoring
 * strategies and allows mock injection for PHPUnit testing.
 *
 * Scoring strategies:
 * - **weightedmean** (default): Average of detector risks weighted by importance.
 * - **maximum**: Uses the highest detector risk.
 * - **sumcap**: Sums all detector risks, capped at 100.
 *
 * For test automation, a mock scorer instance can be injected via
 * {@see self::set_test_instance()}.
 *
 * @package     local_behavioranalytics
 * @category    analytics
 */
class scorer {
    /** @var self|null Optional mock instance for testing. */
    private static ?self $testinstance = null;

    /**
     * Register or clear a mock scorer instance (for testing).
     *
     * @param self|null $instance Optional mock scorer instance or null to clear.
     * @return void
     */
    public static function set_test_instance(?self $instance): void {
        self::$testinstance = $instance;
    }

    /**
     * Return either the test mock instance or a real scorer.
     *
     * @return self Active scorer instance.
     */
    public static function instance(): self {
        return self::$testinstance ?? new self();
    }

    /**
     * Public static entry point used by production code.
     *
     * Delegates to the instance method (mockable in unit tests).
     *
     * @param stdClass $user Moodle user record.
     * @return array{score:int, level:string, findings:array[]} Computed result.
     */
    public static function score_user(stdClass $user): array {
        $instance = self::$testinstance ?? new self();
        return $instance->score_user_instance($user);
    }

    /**
     * Instance-based scoring method (real implementation).
     *
     * Executes all registered detectors for a user and aggregates their
     * outputs using the selected strategy.
     *
     * @param stdClass $user Moodle user record.
     * @return array{score:int, level:string, findings:array[]} Result data.
     */
    public function score_user_instance(stdClass $user): array {
        $findings = registry::run_for_user($user);
        return self::score_user_from_findings($findings);
    }

    /**
     * Compute risk score using configured aggregation strategy.
     *
     * Aggregates detector findings into a final score using one of the
     * supported strategies (`weightedmean`, `maximum`, `sumcap`).
     *
     * @param array $findings Detector result set.
     * @return array{score:int, level:string, findings:array[]} Score summary.
     */
    public static function score_user_from_findings(array $findings): array {
        $weighted = 0.0;
        $sumw = 0.0;
        $details = [];

        foreach ($findings as $f) {
            $ident = (string)($f['ident'] ?? '');
            $risk = max(0, min(100, (int)($f['risk'] ?? 0)));
            $weight = (float)get_config('local_behavioranalytics', $ident . '_weight');
            if (!is_finite($weight) || $weight < 0) {
                $weight = 1.0;
            }

            $contrib = $risk * $weight;
            $weighted += $contrib;
            $sumw += $weight;

            $details[] = [
                'source' => (string)($f['source'] ?? $ident),
                'ident' => $ident,
                'risk' => $risk,
                'weight' => $weight,
                'weighted' => $contrib,
                'message' => (string)($f['message'] ?? ''),
            ];
        }

        // Read the configured scoring strategy.
        $strategy = get_config('local_behavioranalytics', 'scoring_strategy') ?? 'weightedmean';
        $score = 0;

        switch ($strategy) {
            case 'maximum':
                $score = (int)max(array_map(static fn($f) => $f['risk'] ?? 0, $findings ?: [[0]]));
                break;

            case 'sumcap':
                $score = (int)min(array_sum(array_map(static fn($f) => $f['risk'] ?? 0, $findings)), 100);
                break;

            case 'weightedmean':
            default:
                $score = ($sumw > 0) ? (int)round($weighted / $sumw) : 0;
                break;
        }

        $level = self::level_for($score);

        return ['score' => $score, 'level' => $level, 'findings' => $details];
    }

    /**
     * Map a numerical score to a qualitative risk level.
     *
     * @param int $score Risk score (0–100).
     * @return string Localised level string.
     */
    public static function level_for(int $score): string {
        if ($score >= 71) {
            return get_string('risk_level_high', 'local_behavioranalytics');
        }
        if ($score >= 31) {
            return get_string('risk_level_medium', 'local_behavioranalytics');
        }
        return get_string('risk_level_low', 'local_behavioranalytics');
    }
}
