<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Privacy provider for local_behavioranalytics.
 *
 * Defines how the plugin stores and manages user-related data,
 * fulfilling Moodle’s GDPR and privacy API requirements.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\privacy;

use context;
use context_system;
use core_privacy\local\metadata\collection;
use core_privacy\local\request\approved_contextlist;
use core_privacy\local\request\approved_userlist;
use core_privacy\local\request\contextlist;
use core_privacy\local\request\userlist;

/**
 * Privacy provider implementation for local_behavioranalytics.
 *
 * This class describes the user data stored by the plugin (risk profiles)
 * and implements Moodle’s standard privacy interfaces for data export
 * and deletion operations.
 *
 * @package     local_behavioranalytics
 */
class provider implements
    \core_privacy\local\metadata\provider,
    \core_privacy\local\request\plugin\provider,
    \core_privacy\local\request\userlist_provider {
    /**
     * Describe all user data stored by this plugin.
     *
     * @param collection $items The metadata collection to populate.
     * @return collection The updated metadata collection.
     */
    public static function get_metadata(collection $items): collection {
        $items->add_database_table('local_behavioranalytics_profile', [
            'userid'       => 'privacy:metadata:local_behavioranalytics_profile:userid',
            'score'        => 'privacy:metadata:local_behavioranalytics_profile:score',
            'level'        => 'privacy:metadata:local_behavioranalytics_profile:level',
            'findings'     => 'privacy:metadata:local_behavioranalytics_profile:findings',
            'timemodified' => 'privacy:metadata:local_behavioranalytics_profile:timemodified',
        ], 'privacy:metadata:local_behavioranalytics_profile');
        return $items;
    }

    /**
     * Get the list of contexts that contain data for a given user.
     *
     * @param int $userid The user ID to search for.
     * @return contextlist List of contexts containing user data.
     */
    public static function get_contexts_for_userid(int $userid): contextlist {
        $contextlist = new contextlist();
        $contextlist->add_system_context();
        return $contextlist;
    }

    /**
     * Export stored user data for GDPR export requests.
     *
     * Currently not implemented because the plugin only stores
     * analytical summaries (not sensitive personal data).
     *
     * @param approved_contextlist $contextlist List of approved contexts.
     * @return void
     */
    public static function export_user_data(approved_contextlist $contextlist): void {
        // Optional: Implement if export of behaviour data is required.
    }

    /**
     * Delete all user data within a given context.
     *
     * @param context $context The Moodle context object.
     * @return void
     */
    public static function delete_data_for_all_users_in_context(context $context): void {
        global $DB;
        if ($context->contextlevel === CONTEXT_SYSTEM) {
            $DB->delete_records('local_behavioranalytics_profile');
        }
    }

    /**
     * Get the list of users with data in a given context.
     *
     * @param userlist $userlist The user list object to populate.
     * @return void
     */
    public static function get_users_in_context(userlist $userlist): void {
        global $DB;
        if ($userlist->get_context()->contextlevel === CONTEXT_SYSTEM) {
            $userlist->add_users($DB->get_fieldset_sql(
                'SELECT userid FROM {local_behavioranalytics_profile}'
            ));
        }
    }

    /**
     * Delete all stored data for a single user.
     *
     * @param approved_contextlist $contextlist List of approved contexts for the user.
     * @return void
     */
    public static function delete_data_for_user(approved_contextlist $contextlist): void {
        global $DB;
        $userid = $contextlist->get_user()->id;
        $DB->delete_records('local_behavioranalytics_profile', ['userid' => $userid]);
    }

    /**
     * Delete all stored data for multiple users.
     *
     * @param approved_userlist $userlist List of approved users.
     * @return void
     */
    public static function delete_data_for_users(approved_userlist $userlist): void {
        global $DB;
        $userids = $userlist->get_userids();
        if (!empty($userids)) {
            [$in, $params] = $DB->get_in_or_equal($userids, SQL_PARAMS_NAMED);
            $DB->delete_records_select('local_behavioranalytics_profile', "userid $in", $params);
        }
    }
}
