<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Scheduled task that scans users for anomalous behaviour,
 * persists their risk profiles, and executes mitigations.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_behavioranalytics\task;

use core\task\scheduled_task;
use local_behavioranalytics\local\risk\scorer;
use local_behavioranalytics\local\action\registry as action_registry;
use stdClass;

/**
 * Task: Scan user behaviour logs for anomalies and apply mitigations.
 *
 * This task iterates through all active (non-deleted, non-suspended, non-guest)
 * users, computes their behaviour risk profile via {@see scorer}, optionally
 * persists that data, and triggers configured mitigation actions for users
 * whose scores meet or exceed the threshold. Each user is only alerted once
 * per high-risk period (via the `alerted` flag in the `local_behavioranalytics_profile` table).
 */
class scan_anomalies extends scheduled_task {
    /**
     * Get the human-readable name of the scheduled task.
     *
     * @return string Localised task name.
     */
    public function get_name(): string {
        return get_string('task_scan_anomalies', 'local_behavioranalytics');
    }

    /**
     * Execute the anomaly scanning process.
     *
     * @return void
     */
    public function execute(): void {
        global $DB, $CFG;

        // ---------------------------------------------------------------------
        // Configuration values.
        // ---------------------------------------------------------------------
        $persist = (bool)get_config('local_behavioranalytics', 'persistprofiles');
        $threshold = (int)get_config('local_behavioranalytics', 'mitigation_threshold');
        $guestid = $CFG->siteguest ?? 1;

        // ---------------------------------------------------------------------
        // Iterate over all relevant users.
        // ---------------------------------------------------------------------
        $rs = $DB->get_recordset_select(
            'user',
            'deleted = 0 AND suspended = 0 AND id <> :guestid',
            ['guestid' => $guestid],
            'id ASC',
            'id, firstname, lastname, username, suspended, auth'
        );

        foreach ($rs as $user) {
            // Skip service or system accounts if desired.
            if ($user->auth === 'webservice') {
                continue;
            }

            // -----------------------------------------------------------------
            // 1. Compute current risk profile.
            // -----------------------------------------------------------------
            $profile = scorer::score_user($user);

            // -----------------------------------------------------------------
            // 2. Persist to DB (for reporting / continuity).
            // -----------------------------------------------------------------
            $existing = $DB->get_record('local_behavioranalytics_profile', ['userid' => $user->id]);
            $record = (object)[
                'userid' => $user->id,
                'score' => $profile['score'],
                'level' => $profile['level'],
                'findings' => json_encode($profile['findings']),
                'timemodified' => time(),
            ];

            if ($existing) {
                $record->id = $existing->id;
                if ($persist) {
                    $DB->update_record('local_behavioranalytics_profile', $record);
                }
            } else if ($persist) {
                $DB->insert_record('local_behavioranalytics_profile', $record);
            }

            // -----------------------------------------------------------------
            // 3. Mitigation handling with alert deduplication.
            // -----------------------------------------------------------------
            $score = $profile['score'];
            $alerted = $existing->alerted ?? 0;

            if ($score >= $threshold) {
                // Only trigger actions once per high-risk period.
                if (empty($alerted)) {
                    action_registry::execute_selected($user, $score, $profile['level']);
                    if ($persist) {
                        $DB->set_field('local_behavioranalytics_profile', 'alerted', 1, ['userid' => $user->id]);
                    }
                }
            } else {
                // Reset alert flag when user drops below threshold.
                if (!empty($alerted)) {
                    $DB->set_field('local_behavioranalytics_profile', 'alerted', 0, ['userid' => $user->id]);
                }
            }
        }

        $rs->close();
    }
}
