<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Per-user details page (detector breakdown).
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require(__DIR__ . '/../../config.php');

$userid = required_param('id', PARAM_INT);

require_login();
$context = context_system::instance();
require_capability('local/behavioranalytics:view', $context);

$PAGE->set_context($context);
$PAGE->set_url(new moodle_url('/local/behavioranalytics/details.php', ['id' => $userid]));
$PAGE->set_title(get_string('report_details', 'local_behavioranalytics'));

echo $OUTPUT->header();

global $DB;
$user = $DB->get_record('user', ['id' => $userid], '*', MUST_EXIST);

echo $OUTPUT->heading(fullname($user) . ' — ' . get_string('report_details', 'local_behavioranalytics'));

$profile = $DB->get_record('local_behavioranalytics_profile', ['userid' => $userid]);
if (!$profile) {
    echo $OUTPUT->notification(get_string('no_data', 'local_behavioranalytics'), \core\output\notification::NOTIFY_INFO);
    echo html_writer::div(
        $OUTPUT->single_button(
            new moodle_url('/local/behavioranalytics/index.php'),
            get_string('back'),
            'get'
        ),
        'mt-4'
    );
    echo $OUTPUT->footer();
    exit;
}

echo html_writer::tag(
    'p',
    get_string('risk_score', 'local_behavioranalytics') . ': ' .
    (int)$profile->score . ' (' . s($profile->level) . ')'
);

$findings = json_decode((string)$profile->findings, true) ?? [];
$table = new html_table();
$table->head = [
    get_string('detector', 'local_behavioranalytics'),
    get_string('raw_risk', 'local_behavioranalytics'),
    get_string('weight', 'local_behavioranalytics'),
    get_string('weighted', 'local_behavioranalytics'),
    get_string('message', 'local_behavioranalytics'),
];

foreach ($findings as $f) {
    $table->data[] = [
        s((string)($f['source'] ?? $f['ident'] ?? '')),
        (int)($f['risk'] ?? 0),
        format_float((float)($f['weight'] ?? 1.0), 2),
        format_float((float)($f['weighted'] ?? 0.0), 2),
        s((string)($f['message'] ?? '')),
    ];
}

echo html_writer::table($table);

// -----------------------------------------------------------------------------
// Back button
// -----------------------------------------------------------------------------
echo html_writer::div(
    $OUTPUT->single_button(
        new moodle_url('/local/behavioranalytics/index.php'),
        get_string('back'),
        'get'
    ),
    'mt-4'
);

echo $OUTPUT->footer();
