<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Report page: displays top users by anomaly score.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require(__DIR__ . '/../../config.php');

require_login();
$context = context_system::instance();
require_capability('local/behavioranalytics:view', $context);

$PAGE->set_context($context);
$PAGE->set_url(new moodle_url('/local/behavioranalytics/index.php'));
$PAGE->set_title(get_string('report_title', 'local_behavioranalytics'));
$PAGE->set_heading(get_string('report_title', 'local_behavioranalytics'));

echo $OUTPUT->header();

global $DB;

// -----------------------------------------------------------------------------
// Build safe user field list.
// -----------------------------------------------------------------------------
$ufields = \core_user\fields::for_name()
    ->with_userpic()
    ->get_sql('u', false, '', '', false);

// -----------------------------------------------------------------------------
// Compose SQL selecting risk profile + full user info, excluding guest.
// -----------------------------------------------------------------------------
$guestid = $CFG->siteguest ?? 1; // Standard guest user ID fallback to 1.

$sql = "SELECT p.id AS profileid, p.userid, p.score, p.level, {$ufields->selects}
          FROM {local_behavioranalytics_profile} p
          JOIN {user} u ON u.id = p.userid
         WHERE u.deleted = 0
           AND u.id <> :guestid
      ORDER BY p.score DESC, p.timemodified DESC";

$profiles = $DB->get_records_sql($sql, ['guestid' => $guestid], 0, 50);

// -----------------------------------------------------------------------------
// Render results.
// -----------------------------------------------------------------------------
if (!$profiles) {
    echo $OUTPUT->notification(
        get_string('no_data', 'local_behavioranalytics'),
        \core\output\notification::NOTIFY_INFO
    );
} else {
    $table = new html_table();
    $table->head = [
        get_string('user'),
        get_string('risk_score', 'local_behavioranalytics'),
        '',
    ];

    foreach ($profiles as $p) {
        // Ensure $p contains all required name fields for fullname() & user_picture().
        $user = (object)[
            'id' => $p->userid,
            'firstname' => $p->firstname,
            'lastname' => $p->lastname,
            'firstnamephonetic' => $p->firstnamephonetic ?? '',
            'lastnamephonetic' => $p->lastnamephonetic ?? '',
            'middlename' => $p->middlename ?? '',
            'alternatename' => $p->alternatename ?? '',
            'picture' => $p->picture ?? 0,
            'imagealt' => $p->imagealt ?? null,
            'email' => $p->email ?? '',
        ];

        $detailsurl = new moodle_url('/local/behavioranalytics/details.php', ['id' => $p->userid]);

        $table->data[] = [
            $OUTPUT->user_picture($user, ['size' => 35]) . ' ' . fullname($user),
            (int)$p->score . ' (' . s($p->level) . ')',
            html_writer::link($detailsurl, get_string('view_details', 'local_behavioranalytics')),
        ];
    }

    echo html_writer::table($table);
}

echo $OUTPUT->footer();
