<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Admin settings for local_behavioranalytics.
 *
 * @package     local_behavioranalytics
 * @author      Christopher Reimann
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

use local_behavioranalytics\local\detector\registry as detector_registry;
use local_behavioranalytics\local\action\registry as action_registry;

if ($hassiteconfig) {
    // 1️⃣ Add report link under "Site administration → Reports".
    $ADMIN->add('reports', new admin_externalpage(
        'local_behavioranalytics_index',
        get_string('pluginname', 'local_behavioranalytics'),
        new moodle_url('/local/behavioranalytics/index.php'),
        'moodle/site:config'
    ));

    // 2️⃣ Settings page for configuration under "Plugins → Local plugins".
    $settings = new admin_settingpage(
        'local_behavioranalytics',
        get_string('pluginname', 'local_behavioranalytics')
    );

    // -------------------------------------------------------------------------
    // Section: General behaviour
    // -------------------------------------------------------------------------
    $settings->add(new admin_setting_heading(
        'local_behavioranalytics/general',
        get_string('generalsettings', 'local_behavioranalytics'),
        get_string('generalsettings_desc', 'local_behavioranalytics')
    ));

    $settings->add(new admin_setting_configcheckbox(
        'local_behavioranalytics/persistprofiles',
        get_string('persistprofiles', 'local_behavioranalytics'),
        get_string('persistprofiles_desc', 'local_behavioranalytics'),
        1
    ));

    // -------------------------------------------------------------------------
    // Section: Risk scoring
    // -------------------------------------------------------------------------
    $settings->add(new admin_setting_heading(
        'local_behavioranalytics/scoringheading',
        get_string('settings_scoring', 'local_behavioranalytics'),
        get_string('settings_scoring_desc', 'local_behavioranalytics')
    ));

    $settings->add(new admin_setting_configselect(
        'local_behavioranalytics/scoring_strategy',
        get_string('scoringstrategy', 'local_behavioranalytics'),
        get_string('scoringstrategy_desc', 'local_behavioranalytics'),
        'weightedmean',
        [
            'weightedmean' => get_string('scoringstrategy_weightedmean', 'local_behavioranalytics'),
            'maximum'      => get_string('scoringstrategy_maximum', 'local_behavioranalytics'),
            'sumcap'       => get_string('scoringstrategy_sumcap', 'local_behavioranalytics'),
        ]
    ));

    // -------------------------------------------------------------------------
    // Section: Rules (detectors)
    // -------------------------------------------------------------------------
    $settings->add(new admin_setting_heading(
        'local_behavioranalytics/rulesheading',
        get_string('settings_rules', 'local_behavioranalytics'),
        get_string('settings_rules_desc', 'local_behavioranalytics')
    ));

    foreach (detector_registry::all() as $fqcn) {
        // Visual marker/heading for each detector block.
        $ident = defined("$fqcn::IDENT") ? $fqcn::IDENT : strtolower((new \ReflectionClass($fqcn))->getShortName());
        $title = method_exists($fqcn, 'get_name') ? $fqcn::get_name() : $ident;

        $settings->add(new admin_setting_heading(
            'local_behavioranalytics/rule_' . $ident . '_heading',
            $title,
            ''
        ));

        if (method_exists($fqcn, 'add_settings')) {
            $fqcn::add_settings($settings);
        }
    }

    // -------------------------------------------------------------------------
    // Section: Mitigation
    // -------------------------------------------------------------------------
    $settings->add(new admin_setting_heading(
        'local_behavioranalytics/mitigationheading',
        get_string('settings_mitigation', 'local_behavioranalytics'),
        get_string('settings_mitigation_desc', 'local_behavioranalytics')
    ));

    $settings->add(new admin_setting_configtext(
        'local_behavioranalytics/mitigation_threshold',
        get_string('mitigation_threshold', 'local_behavioranalytics'),
        get_string('mitigation_threshold_desc', 'local_behavioranalytics'),
        '80',
        PARAM_INT
    ));

    $actions = [];
    foreach (action_registry::all() as $fqcn) {
        $actions[$fqcn::IDENT] = $fqcn::get_name();
    }

    $settings->add(new admin_setting_configmulticheckbox(
        'local_behavioranalytics/enabledactions',
        get_string('enabledactions', 'local_behavioranalytics'),
        get_string('enabledactions_desc', 'local_behavioranalytics'),
        [],
        $actions
    ));

    // Allow actions to inject their own settings dynamically.
    foreach (action_registry::all() as $fqcn) {
        if (method_exists($fqcn, 'add_settings')) {
            $fqcn::add_settings($settings);
        }
    }

    // Register final settings page.
    $ADMIN->add('localplugins', $settings);
}
