<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * PHPUnit tests for the activity burst detector.
 *
 * @package     local_behavioranalytics
 * @category    test
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\tests;

use advanced_testcase;
use local_behavioranalytics\local\detector\activity_burst;

/**
 * Tests for {@see local_behavioranalytics\local\detector\activity_burst}.
 *
 * Verifies detection of event bursts within a configurable time window.
 *
 * @covers \local_behavioranalytics\local\detector\activity_burst
 */
final class activity_burst_detector_test extends advanced_testcase {
    /**
     * Reset database and environment before each test.
     *
     * @return void
     */
    protected function setUp(): void {
        $this->resetAfterTest();
        parent::setUp();
    }

    /**
     * Insert simulated logstore events for a user.
     *
     * @param int $userid Moodle user ID.
     * @param int $count Number of events to create.
     * @param int $windowseconds Total time span (seconds) covered by events.
     * @return void
     */
    private function insert_events(int $userid, int $count, int $windowseconds): void {
        global $DB;
        $now = time();

        for ($i = 0; $i < $count; $i++) {
            $record = (object)[
                'eventname' => '\core\event\something_happened',
                'component' => 'core',
                'action' => 'viewed',
                'target' => 'course',
                'contextid' => 1,
                'contextlevel' => 50,
                'contextinstanceid' => 1,
                'userid' => $userid,
                'timecreated' => $now - (int)($i * ($windowseconds / max(1, $count))),
                'origin' => 'web',
                'ip' => '127.0.0.1',
                'crud' => 'r', // ✅ Required by schema.
                'edulevel' => \core\event\base::LEVEL_OTHER, // ✅ Required, non-null.
            ];
            $DB->insert_record('logstore_standard_log', $record);
        }
    }

    /**
     * Verify that users with no activity produce no findings.
     *
     * @return void
     */
    public function test_no_activity_returns_empty(): void {
        $user = self::getDataGenerator()->create_user();
        $detector = new activity_burst();
        $this->assertEmpty($detector->detect($user));
    }

    /**
     * Verify that activity below the threshold does not trigger a risk finding.
     *
     * @return void
     */
    public function test_below_threshold_returns_empty(): void {
        $user = self::getDataGenerator()->create_user();
        set_config('activityburst_events', 10, 'local_behavioranalytics');
        set_config('activityburst_window', 60, 'local_behavioranalytics');

        $this->insert_events((int)$user->id, 5, 60);
        $detector = new activity_burst();

        $this->assertEmpty($detector->detect($user));
    }

    /**
     * Verify that exceeding the configured threshold triggers a finding.
     *
     * @return void
     */
    public function test_exceeding_threshold_triggers_alert(): void {
        $user = self::getDataGenerator()->create_user();
        set_config('activityburst_events', 5, 'local_behavioranalytics');
        set_config('activityburst_window', 60, 'local_behavioranalytics');

        $this->insert_events((int)$user->id, 6, 30);
        $detector = new activity_burst();
        $result = $detector->detect($user);

        $this->assertNotEmpty($result);
        $this->assertGreaterThanOrEqual(50, $result[0]['risk']);
        $this->assertStringContainsString('events', $result[0]['message']);
    }
}
