<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Unit tests for the failed_logins detector.
 *
 * @package     local_behavioranalytics
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\tests;

use advanced_testcase;
use local_behavioranalytics\local\detector\failed_logins;

/**
 * Tests for {@see local_behavioranalytics\local\detector\failed_logins}.
 */
final class failed_logins_detector_test extends advanced_testcase {
    /**
     * Reset after each test.
     *
     * @return void
     */
    protected function setUp(): void {
        $this->resetAfterTest();
        parent::setUp();
    }

    /**
     * Insert a synthetic log record into the standard logstore.
     *
     * Accepts int|string for $userid because DB drivers in tests may return string IDs.
     *
     * @param string     $eventname    Fully-qualified event name (e.g. '\core\event\user_login_failed').
     * @param int|string $userid       Moodle user ID (cast to int internally).
     * @param string     $username     Username referenced by the event payload.
     * @param int        $timecreated  Unix timestamp of the event.
     * @return void
     */
    private function insert_log(string $eventname, int|string $userid, string $username, int $timecreated): void {
        global $DB;

        $record = (object)[
            'eventname' => $eventname,
            'component' => 'core',
            'action' => ($eventname === '\core\event\user_login_failed') ? 'failed' : 'loggedin',
            'target' => 'user_login',
            'objecttable' => '',
            'objectid' => null,
            'crud' => 'r',
            'edulevel' => 0,
            'contextid' => 1,
            'contextlevel' => 10,
            'contextinstanceid' => 1,
            'userid' => (int)$userid,
            'anonymous' => 0,
            'relateduserid' => null,
            'other' => json_encode(['username' => $username, 'reason' => 2]),
            'timecreated' => $timecreated,
            'origin' => 'web',
            'ip' => '127.0.0.1',
        ];

        $DB->insert_record('logstore_standard_log', $record);
    }

    /**
     * No failed logins → no findings.
     *
     * @return void
     */
    public function test_no_failures_returns_empty(): void {
        $user = self::getDataGenerator()->create_user(['username' => 'alice']);
        $detector = new failed_logins();
        $result = $detector->detect($user);
        $this->assertEmpty($result);
    }

    /**
     * Below threshold → no alert.
     *
     * @return void
     */
    public function test_below_threshold_no_alert(): void {
        $user = self::getDataGenerator()->create_user(['username' => 'bob']);

        set_config('failedlogins_threshold', 5, 'local_behavioranalytics');
        set_config('failedlogins_window', 600, 'local_behavioranalytics');

        $time = time();
        for ($i = 0; $i < 3; $i++) {
            $this->insert_log('\core\event\user_login_failed', (int)$user->id, (string)$user->username, $time - ($i * 10));
        }

        $detector = new failed_logins();
        $result = $detector->detect($user);

        $this->assertEmpty($result, 'Below threshold should not trigger risk.');
    }

    /**
     * Exceeding threshold raises risk.
     *
     * @return void
     */
    public function test_exceeding_threshold_triggers_alert(): void {
        $user = self::getDataGenerator()->create_user(['username' => 'charlie']);

        set_config('failedlogins_threshold', 3, 'local_behavioranalytics');
        set_config('failedlogins_window', 600, 'local_behavioranalytics');

        $time = time();
        for ($i = 0; $i < 4; $i++) {
            $this->insert_log('\core\event\user_login_failed', (int)$user->id, (string)$user->username, $time - ($i * 10));
        }

        $detector = new failed_logins();
        $result = $detector->detect($user);

        $this->assertNotEmpty($result);
        $this->assertGreaterThanOrEqual(40, $result[0]['risk']);
        $this->assertStringContainsString('failed', $result[0]['message']);
    }

    /**
     * Recent successful login resets failure count.
     *
     * @return void
     */
    public function test_successful_login_resets_failures(): void {
        $user = self::getDataGenerator()->create_user(['username' => 'diana']);

        set_config('failedlogins_threshold', 3, 'local_behavioranalytics');
        set_config('failedlogins_window', 600, 'local_behavioranalytics');

        $time = time();

        // Insert failures far enough back to be within window.
        for ($i = 0; $i < 4; $i++) {
            $this->insert_log('\core\event\user_login_failed', (int)$user->id, (string)$user->username, $time - 600 + ($i * 10));
        }

        // Then a successful login (should reset).
        $this->insert_log('\core\event\user_loggedin', (int)$user->id, (string)$user->username, $time - 30);

        $detector = new failed_logins();
        $result = $detector->detect($user);

        $this->assertEmpty($result, 'Failures before a successful login should not count.');
    }

    /**
     * Guest and webservice users are ignored.
     *
     * @return void
     */
    public function test_guest_and_webservice_users_skipped(): void {
        $guest = \core_user::get_user_by_username('guest');
        $web = self::getDataGenerator()->create_user(['username' => 'apiuser', 'auth' => 'webservice']);

        $detector = new failed_logins();

        $this->assertEmpty($detector->detect($guest), 'Guest user must be skipped.');
        $this->assertEmpty($detector->detect($web), 'Webservice user must be skipped.');
    }
}
