<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * PHPUnit tests for the external web service get_user_risk.
 *
 * @package     local_behavioranalytics
 * @category    test
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\tests;

use advanced_testcase;
use local_behavioranalytics\external\get_user_risk;

/**
 * Tests for {@see \local_behavioranalytics\external\get_user_risk}.
 *
 * @covers \local_behavioranalytics\external\get_user_risk
 *
 * @runTestsInSeparateProcesses
 */
final class get_user_risk_test extends advanced_testcase {
    /**
     * Reset the environment between tests.
     *
     * @return void
     */
    protected function setUp(): void {
        $this->resetAfterTest(true);
        parent::setUp();
    }

    /**
     * Detect the actual profile table name (supports renamed tables).
     *
     * @return string
     */
    private function profile_table(): string {
        global $DB;
        $manager = $DB->get_manager();

        foreach (
            [
            'local_behavioranalytics_profile',
            'local_ba_profile',
            ] as $candidate
        ) {
            if ($manager->table_exists($candidate)) {
                return $candidate;
            }
        }

        foreach ($DB->get_tables() as $t) {
            if (preg_match('#^local_.*(ba|behavioranalytics).*profile#', $t)) {
                return $t;
            }
        }

        $this->fail('Profile table not found. Adjust test to your actual table name.');
    }

    /**
     * When a cached profile exists, the service must return that record.
     *
     * @return void
     */
    public function test_returns_cached_profile(): void {
        global $DB, $CFG;

        require_once($CFG->libdir . '/externallib.php');
        $this->setAdminUser();

        $user = self::getDataGenerator()->create_user();
        $table = $this->profile_table();

        $DB->insert_record($table, (object)[
            'userid' => $user->id,
            'score' => 42,
            'level' => 'Medium',
            'findings' => json_encode([]),
            'timemodified' => time(),
        ]);

        // ✅ Explicit integer cast avoids TypeError.
        $result = get_user_risk::execute((int)$user->id);

        $this->assertIsArray($result);
        $this->assertSame(42, $result['score']);
        $this->assertSame('Medium', $result['level']);
    }

    /**
     * When no profile exists, the service must compute a new one.
     *
     * @return void
     */
    public function test_computes_fresh_profile_when_none_exists(): void {
        global $CFG;

        require_once($CFG->libdir . '/externallib.php');
        $this->setAdminUser();

        $user = self::getDataGenerator()->create_user();

        // ✅ Cast ensures correct parameter type.
        $result = get_user_risk::execute((int)$user->id);

        $this->assertIsArray($result);
        $this->assertArrayHasKey('score', $result);
        $this->assertArrayHasKey('level', $result);
        $this->assertGreaterThanOrEqual(0, $result['score']);
        $this->assertIsString($result['level']);
    }
}
