<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * PHPUnit tests for the kill_sessions mitigation action.
 *
 * @package     local_behavioranalytics
 * @category    test
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\tests;

use advanced_testcase;
use local_behavioranalytics\local\action\kill_sessions;

/**
 * Tests for {@see local_behavioranalytics\local\action\kill_sessions}.
 *
 * Verifies that executing this action removes all user sessions
 * (via Moodle’s core session manager).
 *
 * @covers \local_behavioranalytics\local\action\kill_sessions
 */
final class kill_sessions_test extends advanced_testcase {
    /**
     * Ensure that execute() destroys all user sessions in the database.
     *
     * @return void
     */
    public function test_kill_sessions_executes_destroy(): void {
        global $DB;

        $this->resetAfterTest(true);
        $user = self::getDataGenerator()->create_user();

        // Manually insert fake session records.
        $DB->insert_record('sessions', [
            'sid' => 'sess1',
            'userid' => $user->id,
            'state' => 0,
            'timecreated' => time(),
            'timemodified' => time(),
            'firstip' => '127.0.0.1',
            'lastip' => '127.0.0.1',
        ]);
        $DB->insert_record('sessions', [
            'sid' => 'sess2',
            'userid' => $user->id,
            'state' => 0,
            'timecreated' => time(),
            'timemodified' => time(),
            'firstip' => '127.0.0.1',
            'lastip' => '127.0.0.1',
        ]);

        $this->assertTrue($DB->record_exists('sessions', ['userid' => $user->id]));

        // Execute action.
        $action = new kill_sessions();
        $action->execute($user, 95, 'Critical');

        // Verify that all sessions for this user were removed.
        $this->assertFalse($DB->record_exists('sessions', ['userid' => $user->id]));
    }
}
