<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * PHPUnit tests for the unusual login time detector.
 *
 * @package     local_behavioranalytics
 * @category    test
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\tests;

use advanced_testcase;
use local_behavioranalytics\local\detector\login_time;

/**
 * Tests for {@see local_behavioranalytics\local\detector\login_time}.
 *
 * Validates detection of atypical login time ratios, including
 * correct handling of night window wrapping across midnight.
 *
 * @covers \local_behavioranalytics\local\detector\login_time
 */
final class login_time_detector_test extends advanced_testcase {
    /**
     * Reset after each test.
     *
     * @return void
     */
    protected function setUp(): void {
        $this->resetAfterTest();
        parent::setUp();
    }

    /**
     * Helper: Insert a simulated login event for a given hour.
     *
     * @param int $userid Moodle user ID.
     * @param int $hour Hour of day (0–23).
     * @return void
     */
    private function insert_login_at_hour(int $userid, int $hour): void {
        global $DB;

        $timestamp = strtotime("today {$hour}:00:00") ?: time();
        $record = (object)[
            'eventname' => '\core\event\user_loggedin',
            'component' => 'core',
            'action' => 'loggedin',
            'target' => 'user_login',
            'contextid' => 1,
            'contextlevel' => 10,
            'contextinstanceid' => 1,
            'userid' => $userid,
            'timecreated' => $timestamp,
            'origin' => 'web',
            'ip' => '127.0.0.1',
            'crud' => 'r',
            'edulevel' => \core\event\base::LEVEL_OTHER, // ✅ Non-null required.
        ];

        $DB->insert_record('logstore_standard_log', $record);
    }

    /**
     * Verify that no logins return an empty result.
     *
     * @return void
     */
    public function test_no_logins_returns_empty(): void {
        $user = self::getDataGenerator()->create_user();
        $detector = new login_time();
        $this->assertEmpty($detector->detect($user));
    }

    /**
     * Verify that primarily daytime logins do not trigger a risk.
     *
     * @return void
     */
    public function test_daytime_logins_low_risk(): void {
        $user = self::getDataGenerator()->create_user();
        set_config('nightstarthour', 22, 'local_behavioranalytics');
        set_config('nightendhour', 4, 'local_behavioranalytics');

        foreach (range(9, 17) as $hour) {
            $this->insert_login_at_hour((int)$user->id, $hour);
        }

        $detector = new login_time();
        $this->assertEmpty($detector->detect($user));
    }

    /**
     * Verify that a majority of night logins trigger a risk finding.
     *
     * @return void
     */
    public function test_majority_night_logins_triggers_risk(): void {
        $user = self::getDataGenerator()->create_user();
        set_config('nightstarthour', 22, 'local_behavioranalytics');
        set_config('nightendhour', 4, 'local_behavioranalytics');

        // Eight of ten logins occur during the night window.
        foreach ([23, 0, 1, 2, 3, 23, 0, 1] as $hour) {
            $this->insert_login_at_hour((int)$user->id, $hour);
        }
        foreach ([10, 11] as $hour) {
            $this->insert_login_at_hour((int)$user->id, $hour);
        }

        $detector = new login_time();
        $result = $detector->detect($user);

        $this->assertNotEmpty($result);
        $this->assertGreaterThan(60, $result[0]['risk']);
        $this->assertStringContainsString('night', strtolower($result[0]['message']));
    }
}
