<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * PHPUnit tests for the privacy provider of the plugin.
 *
 * @package     local_behavioranalytics
 * @category    test
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\tests;

use advanced_testcase;
use context_system;
use core_privacy\local\metadata\collection;
use core_privacy\local\request\approved_userlist;
use local_behavioranalytics\privacy\provider;

/**
 * Tests for {@see \local_behavioranalytics\privacy\provider}.
 *
 * Focuses on correct metadata registration and user data deletion.
 *
 * @covers \local_behavioranalytics\privacy\provider
 */
final class privacy_provider_test extends advanced_testcase {
    /**
     * Reset the environment between tests.
     *
     * @return void
     */
    protected function setUp(): void {
        $this->resetAfterTest(true);
        parent::setUp();
    }

    /**
     * Detect the current behavior analytics profile table name.
     *
     * Supports renamed variants (e.g., `local_behavioranalytics_profile`).
     *
     * @return string
     */
    private function profile_table(): string {
        global $DB;
        $manager = $DB->get_manager();

        foreach (
            [
            'local_behavioranalytics_profile',
            'local_ba_profile',
            ] as $candidate
        ) {
            if ($manager->table_exists($candidate)) {
                return $candidate;
            }
        }

        foreach ($DB->get_tables() as $t) {
            if (preg_match('#^local_.*(ba|behavioranalytics).*profile#', $t)) {
                return $t;
            }
        }

        $this->fail('Profile table not found. Adjust test to your actual table name.');
    }

    /**
     * Metadata definition should include the profile table.
     *
     * @return void
     */
    public function test_get_metadata_contains_profile_table(): void {
        $collection = new collection('local_behavioranalytics');
        $result = provider::get_metadata($collection);

        $this->assertInstanceOf(collection::class, $result);

        // Convert metadata items to a readable array for inspection.
        $items = array_map(static function ($item) {
            return $item->get_name();
        }, $result->get_collection());

        $found = false;
        foreach ($items as $name) {
            if (str_contains($name, 'profile')) {
                $found = true;
                break;
            }
        }

        $this->assertTrue($found, 'Expected metadata to reference a profile table.');
    }

    /**
     * Deleting data for a specific user should remove their profile record.
     *
     * @return void
     */
    public function test_delete_data_for_user_removes_profile(): void {
        global $DB;

        $table = $this->profile_table();

        $user = self::getDataGenerator()->create_user();

        $DB->insert_record($table, (object)[
            'userid' => $user->id,
            'score' => 61,
            'level' => 'Medium',
            'findings' => json_encode([]),
            'timemodified' => time(),
        ]);

        $contextlist = new \core_privacy\local\request\approved_contextlist(
            $user,
            'local_behavioranalytics',
            [context_system::instance()->id]
        );

        provider::delete_data_for_user($contextlist);

        $this->assertFalse(
            $DB->record_exists($table, ['userid' => $user->id]),
            'User profile should have been deleted by privacy provider.'
        );
    }
}
