<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * PHPUnit tests for detector registry.
 *
 * @package     local_behavioranalytics
 * @category    test
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\tests;

use advanced_testcase;
use local_behavioranalytics\local\detector\registry;

/**
 * Tests detector registry discovery and rule loading.
 *
 * @covers \local_behavioranalytics\local\detector\registry
 */
final class registry_detectors_test extends advanced_testcase {
    /**
     * Ensure that at least the built-in detectors are discovered automatically.
     *
     * @return void
     */
    public function test_detector_registry_discovers_builtin_detectors(): void {
        $this->resetAfterTest();

        $detectors = registry::all();

        $this->assertNotEmpty(
            $detectors,
            'Detector registry must discover built-in detectors.'
        );

        $this->assertContains(
            'local_behavioranalytics\\local\\detector\\activity_burst',
            $detectors,
            'activity_burst detector must be discovered automatically.'
        );
    }

    /**
     * Verify that running detectors for a user returns structured results.
     *
     * @return void
     */
    public function test_run_for_user_returns_findings(): void {
        $this->resetAfterTest();
        $user = $this->getDataGenerator()->create_user();

        $findings = registry::run_for_user($user);

        $this->assertIsArray($findings);
        foreach ($findings as $finding) {
            $this->assertArrayHasKey('ident', $finding);
            $this->assertArrayHasKey('risk', $finding);
        }
    }
}
