<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * PHPUnit tests for risk scoring strategies.
 *
 * @package     local_behavioranalytics
 * @category    test
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\tests;

use advanced_testcase;
use local_behavioranalytics\local\risk\scorer;

/**
 * Validates risk scoring aggregation strategies.
 *
 * @covers \local_behavioranalytics\local\risk\scorer
 */
final class scorer_weighting_test extends advanced_testcase {
    /**
     * Validate weighted average risk computation.
     *
     * @return void
     */
    public function test_weighted_average_math_is_correct(): void {
        $this->resetAfterTest();

        set_config('rule1_weight', 1.0, 'local_behavioranalytics');
        set_config('rule2_weight', 3.0, 'local_behavioranalytics');
        set_config('scoring_strategy', 'weightedmean', 'local_behavioranalytics');

        $findings = [
            ['ident' => 'rule1', 'risk' => 20, 'message' => ''],
            ['ident' => 'rule2', 'risk' => 80, 'message' => ''],
        ];

        $result = scorer::score_user_from_findings($findings);

        $expected = (20 * 1.0 + 80 * 3.0) / 4.0;
        $this->assertEquals((int)round($expected), $result['score']);
    }

    /**
     * Validate maximum-risk strategy correctly returns highest detector value.
     *
     * @return void
     */
    public function test_maximum_strategy_returns_highest_risk(): void {
        $this->resetAfterTest();

        set_config('scoring_strategy', 'maximum', 'local_behavioranalytics');

        $findings = [
            ['ident' => 'rule1', 'risk' => 25],
            ['ident' => 'rule2', 'risk' => 80],
            ['ident' => 'rule3', 'risk' => 60],
        ];

        $result = scorer::score_user_from_findings($findings);
        $this->assertEquals(80, $result['score']);
    }

    /**
     * Validate capped sum strategy accumulates multiple risks correctly.
     *
     * @return void
     */
    public function test_sumcap_strategy_accumulates_until_100(): void {
        $this->resetAfterTest();

        set_config('scoring_strategy', 'sumcap', 'local_behavioranalytics');

        $findings = [
            ['ident' => 'rule1', 'risk' => 40],
            ['ident' => 'rule2', 'risk' => 70],
        ];

        $result = scorer::score_user_from_findings($findings);
        $this->assertEquals(100, $result['score'], 'Sum-capped score must not exceed 100.');
    }

    /**
     * Validate zero-weight detectors result in zero score for weighted mean.
     *
     * @return void
     */
    public function test_weighted_average_handles_zero_weights(): void {
        $this->resetAfterTest();

        set_config('scoring_strategy', 'weightedmean', 'local_behavioranalytics');
        set_config('rule1_weight', 0.0, 'local_behavioranalytics');

        $findings = [
            ['ident' => 'rule1', 'risk' => 50],
        ];

        $result = scorer::score_user_from_findings($findings);
        $this->assertEquals(0, $result['score']);
    }
}
