<?php
// This file is part of Moodle - https://moodle.org/.
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * PHPUnit test for the scheduled task scan_anomalies.
 *
 * Verifies that users with risk scores above the threshold
 * trigger mitigation actions and event logging correctly,
 * and that low-risk users are left untouched.
 *
 * @package     local_behavioranalytics
 * @category    test
 * @copyright   2025 Christopher Reimann
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

declare(strict_types=1);

namespace local_behavioranalytics\tests;

use advanced_testcase;
use local_behavioranalytics\task\scan_anomalies;
use local_behavioranalytics\local\risk\scorer;
use local_behavioranalytics\event\mitigation_executed;

/**
 * Test case for the scan_anomalies scheduled task.
 */
final class task_scan_anomalies_test extends advanced_testcase {
    /**
     * Setup the test environment.
     *
     * @return void
     */
    protected function setUp(): void {
        parent::setUp();
        $this->resetAfterTest(true);
        $this->preventResetByRollback();
    }

    /**
     * Ensure that a high-risk user triggers the suspend_user mitigation.
     *
     * @covers \local_behavioranalytics\task\scan_anomalies
     * @return void
     */
    public function test_high_risk_user_triggers_mitigation(): void {
        global $DB;

        // Create a test user.
        $user = $this->getDataGenerator()->create_user(['suspended' => 0]);

        // Enable persistence and mitigation settings.
        set_config('persistprofiles', 1, 'local_behavioranalytics');
        set_config('mitigation_threshold', 50, 'local_behavioranalytics');
        set_config('enabledactions', json_encode(['suspend_user']), 'local_behavioranalytics');

        // Mock the scorer to always return a high risk score.
        $mockscorer = $this->getMockBuilder(\local_behavioranalytics\local\risk\scorer::class)
            ->onlyMethods(['score_user_instance'])
            ->getMock();

        $mockscorer->method('score_user_instance')->willReturn([
            'score' => 90,
            'level' => 'High',
            'findings' => [
                ['ident' => 'mock_detector', 'risk' => 90, 'weight' => 1.0, 'message' => 'Mock high risk'],
            ],
        ]);
        \local_behavioranalytics\local\risk\scorer::set_test_instance($mockscorer);

        // Capture events.
        $sink = $this->redirectEvents();
        $task = new scan_anomalies();
        $task->execute();
        $events = $sink->get_events();
        $sink->close();

        // Validate outcomes.
        $userrecord = $DB->get_record('user', ['id' => $user->id], '*', MUST_EXIST);
        $this->assertEquals(1, $userrecord->suspended, 'User should be suspended by mitigation.');

        // Ensure event fired.
        $found = false;
        foreach ($events as $event) {
            if ($event instanceof mitigation_executed) {
                $found = true;
                break;
            }
        }
        $this->assertTrue($found, 'Mitigation_executed event should be triggered.');

        // Verify persistence entry.
        $profile = $DB->get_record('local_behavioranalytics_profile', ['userid' => $user->id]);
        $this->assertNotEmpty($profile, 'User risk profile should be persisted.');
        $this->assertEquals(90, $profile->score);
    }

    /**
     * Ensure that a low-risk user does not trigger mitigation actions.
     *
     * @covers \local_behavioranalytics\task\scan_anomalies
     * @return void
     */
    public function test_low_risk_user_does_not_trigger_mitigation(): void {
        global $DB;

        // Create a test user.
        $user = $this->getDataGenerator()->create_user(['suspended' => 0]);

        // Same settings as before.
        set_config('persistprofiles', 1, 'local_behavioranalytics');
        set_config('mitigation_threshold', 80, 'local_behavioranalytics');
        set_config('enabledactions', json_encode(['suspend_user']), 'local_behavioranalytics');

        // Mock scorer with low score below threshold.
        $mockscorer = $this->getMockBuilder(\local_behavioranalytics\local\risk\scorer::class)
            ->onlyMethods(['score_user_instance'])
            ->getMock();

        $mockscorer->method('score_user_instance')->willReturn([
            'score' => 40,
            'level' => 'Low',
            'findings' => [
                ['ident' => 'mock_detector', 'risk' => 40, 'weight' => 1.0, 'message' => 'Mock low risk'],
            ],
        ]);
        \local_behavioranalytics\local\risk\scorer::set_test_instance($mockscorer);

        // Capture events.
        $sink = $this->redirectEvents();
        $task = new scan_anomalies();
        $task->execute();
        $events = $sink->get_events();
        $sink->close();

        // User should remain unsuspended.
        $userrecord = $DB->get_record('user', ['id' => $user->id], '*', MUST_EXIST);
        $this->assertEquals(0, $userrecord->suspended, 'Low-risk user should not be suspended.');

        // No mitigation events should be triggered.
        $found = false;
        foreach ($events as $event) {
            if ($event instanceof mitigation_executed) {
                $found = true;
                break;
            }
        }
        $this->assertFalse($found, 'No mitigation_executed event should be triggered for low-risk user.');

        // But risk profile should still be persisted.
        $profile = $DB->get_record('local_behavioranalytics_profile', ['userid' => $user->id]);
        $this->assertNotEmpty($profile, 'User risk profile should be persisted even if no mitigation.');
        $this->assertEquals(40, $profile->score);
    }
}
