<?php
// This file is part of plugin tool_vault - https://lmsvault.io
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace tool_vault\local\restoreactions\upgrade_401;

use tool_vault\api;
use tool_vault\constants;
use tool_vault\site_restore;

/**
 * Class upgrade_401
 *
 * @package    tool_vault
 * @copyright  2024 Marina Glancy
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class upgrade_401 {
    /**
     * Upgrade the restored site to 4.1.2
     *
     * @param site_restore $logger
     * @return void
     */
    public static function upgrade(site_restore $logger) {
        self::upgrade_core($logger);
        self::upgrade_plugins($logger);
        set_config('upgraderunning', 0);
    }

    /**
     * Upgrade core to 4.1.2
     *
     * @param site_restore $logger
     * @return void
     */
    protected static function upgrade_core(site_restore $logger) {
        global $CFG;
        require_once(__DIR__ ."/core.php");

        try {
            tool_vault_401_core_upgrade($CFG->version);
        } catch (\Throwable $t) {
            $logger->add_to_log("Exception executing core upgrade script: ".
               $t->getMessage(), constants::LOGLEVEL_WARNING);
            api::report_error($t);
        }

        set_config('version', 2022112802.00);
        set_config('release', '4.1.2');
        set_config('branch', '401');
    }

    /**
     * Upgrade all standard plugins to 4.1.2
     *
     * @param site_restore $logger
     * @return void
     */
    protected static function upgrade_plugins(site_restore $logger) {
        global $DB;
        $allcurversions = $DB->get_records_menu('config_plugins', ['name' => 'version'], '', 'plugin, value');
        foreach (self::plugin_versions() as $plugin => $version) {
            if (empty($allcurversions[$plugin])) {
                // Standard plugin {$plugin} not found. It will be installed during the full upgrade.
                continue;
            }
            if (file_exists(__DIR__ ."/". $plugin .".php") && \core_component::get_component_directory($plugin)) {
                require_once(__DIR__ ."/". $plugin .".php");
                $pluginshort = preg_replace("/^mod_/", "", $plugin);
                $funcname = "tool_vault_401_xmldb_{$pluginshort}_upgrade";
                try {
                    $funcname($allcurversions[$plugin]);
                } catch (\Throwable $t) {
                    $logger->add_to_log("Exception executing upgrade script for plugin {$plugin}: ".
                        $t->getMessage(), constants::LOGLEVEL_WARNING);
                    api::report_error($t);
                }
            }
            set_config('version', $version, $plugin);
        }
    }

    /**
     * List of standard plugins in 4.1.2 and their exact versions
     *
     * @return array
     */
    protected static function plugin_versions() {
        return [
            "mod_assign" => 2022112801,
            "mod_assignment" => 2022112800,
            "mod_bigbluebuttonbn" => 2022112800,
            "mod_book" => 2022112800,
            "mod_chat" => 2022112800,
            "mod_choice" => 2022112800,
            "mod_data" => 2022112801,
            "mod_feedback" => 2022112800,
            "mod_folder" => 2022112800,
            "mod_forum" => 2022112801,
            "mod_glossary" => 2022112800,
            "mod_h5pactivity" => 2022112801,
            "mod_imscp" => 2022112800,
            "mod_label" => 2022112801,
            "mod_lesson" => 2022112800,
            "mod_lti" => 2022112800,
            "mod_page" => 2022112800,
            "mod_quiz" => 2022112800,
            "mod_resource" => 2022112800,
            "mod_scorm" => 2022112800,
            "mod_survey" => 2022112800,
            "mod_url" => 2022112800,
            "mod_wiki" => 2022112800,
            "mod_workshop" => 2022112800,
            "assignsubmission_comments" => 2022112800,
            "assignsubmission_file" => 2022112800,
            "assignsubmission_onlinetext" => 2022112800,
            "assignfeedback_comments" => 2022112800,
            "assignfeedback_editpdf" => 2022112801,
            "assignfeedback_file" => 2022112800,
            "assignfeedback_offline" => 2022112800,
            "assignment_offline" => 2022112800,
            "assignment_online" => 2022112800,
            "assignment_upload" => 2022112800,
            "assignment_uploadsingle" => 2022112800,
            "booktool_exportimscp" => 2022112800,
            "booktool_importhtml" => 2022112800,
            "booktool_print" => 2022112800,
            "datafield_checkbox" => 2022112800,
            "datafield_date" => 2022112800,
            "datafield_file" => 2022112800,
            "datafield_latlong" => 2022112800,
            "datafield_menu" => 2022112800,
            "datafield_multimenu" => 2022112800,
            "datafield_number" => 2022112800,
            "datafield_picture" => 2022112800,
            "datafield_radiobutton" => 2022112800,
            "datafield_text" => 2022112800,
            "datafield_textarea" => 2022112800,
            "datafield_url" => 2022112800,
            "datapreset_imagegallery" => 2022112800,
            "datapreset_journal" => 2022112800,
            "datapreset_proposals" => 2022112800,
            "datapreset_resources" => 2022112800,
            "forumreport_summary" => 2022112800,
            "ltiservice_basicoutcomes" => 2022112800,
            "ltiservice_gradebookservices" => 2022112800,
            "ltiservice_memberships" => 2022112800,
            "ltiservice_profile" => 2022112800,
            "ltiservice_toolproxy" => 2022112800,
            "ltiservice_toolsettings" => 2022112800,
            "quiz_grading" => 2022112800,
            "quiz_overview" => 2022112800,
            "quiz_responses" => 2022112800,
            "quiz_statistics" => 2022112800,
            "quizaccess_delaybetweenattempts" => 2022112800,
            "quizaccess_ipaddress" => 2022112800,
            "quizaccess_numattempts" => 2022112800,
            "quizaccess_offlineattempts" => 2022112800,
            "quizaccess_openclosedate" => 2022112800,
            "quizaccess_password" => 2022112800,
            "quizaccess_seb" => 2022112800,
            "quizaccess_securewindow" => 2022112800,
            "quizaccess_timelimit" => 2022112800,
            "scormreport_basic" => 2022112800,
            "scormreport_graphs" => 2022112800,
            "scormreport_interactions" => 2022112800,
            "scormreport_objectives" => 2022112800,
            "workshopform_accumulative" => 2022112800,
            "workshopform_comments" => 2022112800,
            "workshopform_numerrors" => 2022112800,
            "workshopform_rubric" => 2022112800,
            "workshopallocation_manual" => 2022112800,
            "workshopallocation_random" => 2022112800,
            "workshopallocation_scheduled" => 2022112800,
            "workshopeval_best" => 2022112800,
            "block_accessreview" => 2022112800,
            "block_activity_modules" => 2022112800,
            "block_activity_results" => 2022112800,
            "block_admin_bookmarks" => 2022112800,
            "block_badges" => 2022112800,
            "block_blog_menu" => 2022112800,
            "block_blog_recent" => 2022112800,
            "block_blog_tags" => 2022112800,
            "block_calendar_month" => 2022112800,
            "block_calendar_upcoming" => 2022112800,
            "block_comments" => 2022112800,
            "block_completionstatus" => 2022112800,
            "block_course_list" => 2022112800,
            "block_course_summary" => 2022112800,
            "block_feedback" => 2022112800,
            "block_globalsearch" => 2022112800,
            "block_glossary_random" => 2022112800,
            "block_html" => 2022112800,
            "block_login" => 2022112800,
            "block_lp" => 2022112800,
            "block_mentees" => 2022112800,
            "block_mnet_hosts" => 2022112800,
            "block_myoverview" => 2022112800,
            "block_myprofile" => 2022112800,
            "block_navigation" => 2022112800,
            "block_news_items" => 2022112800,
            "block_online_users" => 2022112800,
            "block_private_files" => 2022112800,
            "block_recent_activity" => 2022112800,
            "block_recentlyaccessedcourses" => 2022112800,
            "block_recentlyaccesseditems" => 2022112800,
            "block_rss_client" => 2022112800,
            "block_search_forums" => 2022112800,
            "block_section_links" => 2022112800,
            "block_selfcompletion" => 2022112800,
            "block_settings" => 2022112800,
            "block_site_main_menu" => 2022112800,
            "block_social_activities" => 2022112800,
            "block_starredcourses" => 2022112800,
            "block_tag_flickr" => 2022112800,
            "block_tag_youtube" => 2022112800,
            "block_tags" => 2022112800,
            "block_timeline" => 2022112800,
            "qtype_calculated" => 2022112800,
            "qtype_calculatedmulti" => 2022112800,
            "qtype_calculatedsimple" => 2022112800,
            "qtype_ddimageortext" => 2022112800,
            "qtype_ddmarker" => 2022112800,
            "qtype_ddwtos" => 2022112800,
            "qtype_description" => 2022112800,
            "qtype_essay" => 2022112800,
            "qtype_gapselect" => 2022112800,
            "qtype_match" => 2022112800,
            "qtype_missingtype" => 2022112800,
            "qtype_multianswer" => 2022112800,
            "qtype_multichoice" => 2022112800,
            "qtype_numerical" => 2022112800,
            "qtype_random" => 2022112800,
            "qtype_randomsamatch" => 2022112800,
            "qtype_shortanswer" => 2022112800,
            "qtype_truefalse" => 2022112800,
            "qbank_bulkmove" => 2022112800,
            "qbank_columnsortorder" => 2022112800,
            "qbank_comment" => 2022112800,
            "qbank_customfields" => 2022112800,
            "qbank_deletequestion" => 2022112800,
            "qbank_editquestion" => 2022112800,
            "qbank_exportquestions" => 2022112800,
            "qbank_exporttoxml" => 2022112800,
            "qbank_history" => 2022112800,
            "qbank_importquestions" => 2022112800,
            "qbank_managecategories" => 2022112800,
            "qbank_previewquestion" => 2022112800,
            "qbank_statistics" => 2022112800,
            "qbank_tagquestion" => 2022112800,
            "qbank_usage" => 2022112800,
            "qbank_viewcreator" => 2022112800,
            "qbank_viewquestionname" => 2022112800,
            "qbank_viewquestiontext" => 2022112800,
            "qbank_viewquestiontype" => 2022112800,
            "qbehaviour_adaptive" => 2022112800,
            "qbehaviour_adaptivenopenalty" => 2022112800,
            "qbehaviour_deferredcbm" => 2022112800,
            "qbehaviour_deferredfeedback" => 2022112800,
            "qbehaviour_immediatecbm" => 2022112800,
            "qbehaviour_immediatefeedback" => 2022112800,
            "qbehaviour_informationitem" => 2022112800,
            "qbehaviour_interactive" => 2022112800,
            "qbehaviour_interactivecountback" => 2022112800,
            "qbehaviour_manualgraded" => 2022112800,
            "qbehaviour_missing" => 2022112800,
            "qformat_aiken" => 2022112800,
            "qformat_blackboard_six" => 2022112800,
            "qformat_gift" => 2022112800,
            "qformat_missingword" => 2022112800,
            "qformat_multianswer" => 2022112800,
            "qformat_xhtml" => 2022112800,
            "qformat_xml" => 2022112800,
            "filter_activitynames" => 2022112800,
            "filter_algebra" => 2022112800,
            "filter_data" => 2022112800,
            "filter_displayh5p" => 2022112800,
            "filter_emailprotect" => 2022112800,
            "filter_emoticon" => 2022112800,
            "filter_glossary" => 2022112800,
            "filter_mathjaxloader" => 2022112800,
            "filter_mediaplugin" => 2022112800,
            "filter_multilang" => 2022112800,
            "filter_tex" => 2022112800,
            "filter_tidy" => 2022112800,
            "filter_urltolink" => 2022112800,
            "editor_atto" => 2022112800,
            "editor_textarea" => 2022112800,
            "editor_tiny" => 2022112800,
            "editor_tinymce" => 2022112800,
            "atto_accessibilitychecker" => 2022112800,
            "atto_accessibilityhelper" => 2022112800,
            "atto_align" => 2022112800,
            "atto_backcolor" => 2022112800,
            "atto_bold" => 2022112800,
            "atto_charmap" => 2022112800,
            "atto_clear" => 2022112800,
            "atto_collapse" => 2022112800,
            "atto_emojipicker" => 2022112800,
            "atto_emoticon" => 2022112800,
            "atto_equation" => 2022112800,
            "atto_fontcolor" => 2022112800,
            "atto_h5p" => 2022112800,
            "atto_html" => 2022112800,
            "atto_image" => 2022112800,
            "atto_indent" => 2022112800,
            "atto_italic" => 2022112800,
            "atto_link" => 2022112800,
            "atto_managefiles" => 2022112800,
            "atto_media" => 2022112800,
            "atto_noautolink" => 2022112800,
            "atto_orderedlist" => 2022112800,
            "atto_recordrtc" => 2022112800,
            "atto_rtl" => 2022112800,
            "atto_strike" => 2022112800,
            "atto_subscript" => 2022112800,
            "atto_superscript" => 2022112800,
            "atto_table" => 2022112800,
            "atto_title" => 2022112800,
            "atto_underline" => 2022112800,
            "atto_undo" => 2022112800,
            "atto_unorderedlist" => 2022112800,
            "tiny_accessibilitychecker" => 2022112800,
            "tiny_autosave" => 2022112800,
            "tiny_equation" => 2022112800,
            "tiny_h5p" => 2022112800,
            "tiny_media" => 2022112800,
            "tiny_recordrtc" => 2022112800,
            "tinymce_ctrlhelp" => 2022112800,
            "tinymce_managefiles" => 2022112800,
            "tinymce_moodleemoticon" => 2022112800,
            "tinymce_moodleimage" => 2022112800,
            "tinymce_moodlemedia" => 2022112800,
            "tinymce_moodlenolink" => 2022112800,
            "tinymce_pdw" => 2022112800,
            "tinymce_spellchecker" => 2022112800,
            "tinymce_wrap" => 2022112800,
            "enrol_category" => 2022112800,
            "enrol_cohort" => 2022112800,
            "enrol_database" => 2022112800,
            "enrol_fee" => 2022112800,
            "enrol_flatfile" => 2022112800,
            "enrol_guest" => 2022112800,
            "enrol_imsenterprise" => 2022112800,
            "enrol_ldap" => 2022112800,
            "enrol_lti" => 2022112800,
            "enrol_manual" => 2022112800,
            "enrol_meta" => 2022112800,
            "enrol_mnet" => 2022112800,
            "enrol_paypal" => 2022112800,
            "enrol_self" => 2022112800,
            "auth_cas" => 2022112800,
            "auth_db" => 2022112800,
            "auth_email" => 2022112800,
            "auth_ldap" => 2022112800,
            "auth_lti" => 2022112800,
            "auth_manual" => 2022112800,
            "auth_mnet" => 2022112800,
            "auth_nologin" => 2022112800,
            "auth_none" => 2022112800,
            "auth_oauth2" => 2022112800,
            "auth_shibboleth" => 2022112800,
            "auth_webservice" => 2022112800,
            "tool_admin_presets" => 2022112800,
            "tool_analytics" => 2022112800,
            "tool_availabilityconditions" => 2022112800,
            "tool_behat" => 2022112802,
            "tool_brickfield" => 2022112800,
            "tool_capability" => 2022112800,
            "tool_cohortroles" => 2022112801,
            "tool_componentlibrary" => 2022112800,
            "tool_customlang" => 2022112800,
            "tool_dataprivacy" => 2022112800,
            "tool_dbtransfer" => 2022112800,
            "tool_filetypes" => 2022112800,
            "tool_generator" => 2022112800,
            "tool_httpsreplace" => 2022112800,
            "tool_innodb" => 2022112800,
            "tool_installaddon" => 2022112800,
            "tool_langimport" => 2022112800,
            "tool_licensemanager" => 2022112800,
            "tool_log" => 2022112800,
            "tool_lp" => 2022112800,
            "tool_lpimportcsv" => 2022112800,
            "tool_lpmigrate" => 2022112800,
            "tool_messageinbound" => 2022112800,
            "tool_mobile" => 2022112800,
            "tool_monitor" => 2022112800,
            "tool_moodlenet" => 2022112800,
            "tool_multilangupgrade" => 2022112800,
            "tool_oauth2" => 2022112800,
            "tool_phpunit" => 2022112800,
            "tool_policy" => 2022112800,
            "tool_profiling" => 2022112800,
            "tool_recyclebin" => 2022112800,
            "tool_replace" => 2022112800,
            "tool_spamcleaner" => 2022112800,
            "tool_task" => 2022112800,
            "tool_templatelibrary" => 2022112800,
            "tool_unsuproles" => 2022112800,
            "tool_uploadcourse" => 2022112800,
            "tool_uploaduser" => 2022112800,
            "tool_usertours" => 2022112800,
            "tool_xmldb" => 2022112800,
            "logstore_database" => 2022112800,
            "logstore_legacy" => 2022112800,
            "logstore_standard" => 2022112800,
            "antivirus_clamav" => 2022112800,
            "availability_completion" => 2022112800,
            "availability_date" => 2022112800,
            "availability_grade" => 2022112800,
            "availability_group" => 2022112800,
            "availability_grouping" => 2022112800,
            "availability_profile" => 2022112800,
            "calendartype_gregorian" => 2022112800,
            "customfield_checkbox" => 2022112800,
            "customfield_date" => 2022112800,
            "customfield_select" => 2022112800,
            "customfield_text" => 2022112800,
            "customfield_textarea" => 2022112800,
            "message_airnotifier" => 2022112800,
            "message_email" => 2022112800,
            "message_popup" => 2022112800,
            "media_html5audio" => 2022112800,
            "media_html5video" => 2022112800,
            "media_videojs" => 2022112800,
            "media_vimeo" => 2022112800,
            "media_youtube" => 2022112800,
            "format_singleactivity" => 2022112800,
            "format_social" => 2022112800,
            "format_topics" => 2022112801,
            "format_weeks" => 2022112801,
            "dataformat_csv" => 2022112800,
            "dataformat_excel" => 2022112800,
            "dataformat_html" => 2022112800,
            "dataformat_json" => 2022112800,
            "dataformat_ods" => 2022112800,
            "dataformat_pdf" => 2022112800,
            "profilefield_checkbox" => 2022112800,
            "profilefield_datetime" => 2022112800,
            "profilefield_menu" => 2022112800,
            "profilefield_social" => 2022112800,
            "profilefield_text" => 2022112800,
            "profilefield_textarea" => 2022112800,
            "report_backups" => 2022112800,
            "report_competency" => 2022112800,
            "report_completion" => 2022112800,
            "report_configlog" => 2022112800,
            "report_courseoverview" => 2022112800,
            "report_eventlist" => 2022112800,
            "report_infectedfiles" => 2022112800,
            "report_insights" => 2022112800,
            "report_log" => 2022112800,
            "report_loglive" => 2022112800,
            "report_outline" => 2022112800,
            "report_participation" => 2022112800,
            "report_performance" => 2022112800,
            "report_progress" => 2022112800,
            "report_questioninstances" => 2022112800,
            "report_security" => 2022112800,
            "report_stats" => 2022112800,
            "report_status" => 2022112800,
            "report_usersessions" => 2022112800,
            "gradeexport_ods" => 2022112800,
            "gradeexport_txt" => 2022112800,
            "gradeexport_xls" => 2022112800,
            "gradeexport_xml" => 2022112800,
            "gradeimport_csv" => 2022112800,
            "gradeimport_direct" => 2022112800,
            "gradeimport_xml" => 2022112800,
            "gradereport_grader" => 2022112800,
            "gradereport_history" => 2022112800,
            "gradereport_outcomes" => 2022112800,
            "gradereport_overview" => 2022112800,
            "gradereport_singleview" => 2022112800,
            "gradereport_summary" => 2022112800,
            "gradereport_user" => 2022112800,
            "gradingform_guide" => 2022112800,
            "gradingform_rubric" => 2022112800,
            "mlbackend_php" => 2022112800,
            "mlbackend_python" => 2022112800,
            "mnetservice_enrol" => 2022112800,
            "webservice_rest" => 2022112800,
            "webservice_soap" => 2022112800,
            "repository_areafiles" => 2022112800,
            "repository_contentbank" => 2022112800,
            "repository_coursefiles" => 2022112800,
            "repository_dropbox" => 2022112800,
            "repository_equella" => 2022112800,
            "repository_filesystem" => 2022112800,
            "repository_flickr" => 2022112800,
            "repository_flickr_public" => 2022112800,
            "repository_googledocs" => 2022112800,
            "repository_local" => 2022112800,
            "repository_merlot" => 2022112800,
            "repository_nextcloud" => 2022112800,
            "repository_onedrive" => 2022112800,
            "repository_recent" => 2022112800,
            "repository_s3" => 2022112800,
            "repository_upload" => 2022112800,
            "repository_url" => 2022112800,
            "repository_user" => 2022112800,
            "repository_webdav" => 2022112800,
            "repository_wikimedia" => 2022112800,
            "repository_youtube" => 2022112800,
            "portfolio_download" => 2022112800,
            "portfolio_flickr" => 2022112800,
            "portfolio_googledocs" => 2022112800,
            "portfolio_mahara" => 2022112800,
            "search_simpledb" => 2022112800,
            "search_solr" => 2022112800,
            "cachestore_apcu" => 2022112800,
            "cachestore_file" => 2022112800,
            "cachestore_memcached" => 2022112800,
            "cachestore_mongodb" => 2022112801,
            "cachestore_redis" => 2022112800,
            "cachestore_session" => 2022112800,
            "cachestore_static" => 2022112800,
            "cachelock_file" => 2022112800,
            "fileconverter_googledrive" => 2022112800,
            "fileconverter_unoconv" => 2022112800,
            "contenttype_h5p" => 2022112800,
            "theme_boost" => 2022112800,
            "theme_classic" => 2022112800,
            "h5plib_v124" => 2022112800,
            "paygw_paypal" => 2022112800,
        ];
    }
}
