# File Name: Moodle-EntraID-Script.ps1
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the GPLv3 License.

#Requires -Version 7.0

[CmdletBinding()]
param()

# Initialize
$ErrorActionPreference = 'Stop'
$ProgressPreference = 'Continue'
$InformationPreference = 'Continue'

# Helper Functions
function Write-StatusMessage {
    param(
        [string]$Message,
        [string]$Color = 'Green'
    )
    Write-Host $Message -ForegroundColor $Color
}

function Invoke-WithRetry {
    param(
        [Parameter(Mandatory)]
        [scriptblock]$ScriptBlock,
        [int]$MaxAttempts = 5,
        [int]$DelaySeconds = 3,
        [string]$OperationName = "Operation"
    )
    
    for ($attempt = 1; $attempt -le $MaxAttempts; $attempt++) {
        try {
            return & $ScriptBlock
        }
        catch {
            if ($attempt -eq $MaxAttempts) {
                Write-StatusMessage "$OperationName failed after $MaxAttempts attempts: $_" -Color Red
                throw
            }
            Write-StatusMessage "Attempt $attempt of $MaxAttempts failed. Retrying in $DelaySeconds seconds..." -Color Yellow
            Start-Sleep -Seconds $DelaySeconds
        }
    }
}

function Import-RequiredModules {
    Write-StatusMessage "Installing and importing required modules..." -Color Cyan
    
    $requiredModules = @(
        'Microsoft.Graph.Applications',
        'Microsoft.Graph.Users',
        'Microsoft.Graph.Identity.SignIns'
    )
    
    foreach ($module in $requiredModules) {
        # Check if module is installed
        $currentModule = Get-Module -ListAvailable -Name $module | Sort-Object Version -Descending | Select-Object -First 1
        
        # Get latest version from PSGallery
        $latestVersion = (Find-Module -Name $module).Version
        
        if (-not $currentModule) {
            # Module not installed, install it
            Write-StatusMessage "Installing module $module..." -Color Yellow
            Install-Module -Name $module -Scope CurrentUser -Force -AllowClobber
        }
        elseif ($currentModule.Version -lt $latestVersion) {
            # Module needs update
            Write-StatusMessage "Updating module $module from version $($currentModule.Version) to $latestVersion..." -Color Yellow
            Update-Module -Name $module -Force
        }
        
        # Import the module
        Import-Module -Name $module -Force
    }
    
    Write-StatusMessage "Module installation completed"
}

function Get-Resources {
    [OutputType([Microsoft.Graph.PowerShell.Models.MicrosoftGraphRequiredResourceAccess[]])]
    param()
    
    $outputArray = @()
    $jsonPath = Join-Path (Get-Location) 'Json\permissions.json'
    
    try {
        $jsonContent = Get-Content $jsonPath -Raw
        $jsonObj = $jsonContent | ConvertFrom-Json
        
        foreach ($resource in $jsonObj.requiredResourceAccess) {
            $reqResourceAccess = [Microsoft.Graph.PowerShell.Models.MicrosoftGraphRequiredResourceAccess]::new()
            $reqResourceAccess.ResourceAppId = $resource.resourceAppId
            
            $resourceAccessArray = $resource.resourceAccess | ForEach-Object {
                $access = [Microsoft.Graph.PowerShell.Models.MicrosoftGraphResourceAccess]::new()
                $access.Id = $_.id
                $access.Type = $_.type
                $access
            }
            
            $reqResourceAccess.ResourceAccess = $resourceAccessArray
            $outputArray += $reqResourceAccess
        }
    }
    catch {
        Write-StatusMessage "Failed to process permissions.json: $_" -Color Red
        throw
    }
    
    return $outputArray
}

function Test-SecureUrl {
    param(
        [Parameter(Mandatory)]
        [string]$Url
    )
    return $Url -match "^https:\/\/(www\.)?[-a-zA-Z0-9@:%._\+~#=]{1,256}\.[a-zA-Z0-9()]{1,6}\b([-a-zA-Z0-9()@:%_\+.~#?&//=]*)$"
}

function Get-MoodleUrl {
    do {
        $url = Read-Host -Prompt "Enter the URL of your Moodle server (ex: https://www.moodleserver.com)"
        if (-not(Test-SecureUrl $url)) {
            Write-StatusMessage "Invalid URL. Please enter a valid HTTPS URL." -Color Red
            continue
        }
        return $url.TrimEnd('/') + '/'
    } while ($true)
}

function Grant-AdminConsent {
    param(
        [Parameter(Mandatory)]
        [string]$ApplicationId
    )
    
    $title = "Admin Consent Required"
    $question = "Do you want to grant admin consent for the application?"
    
    $choices = @(
        [System.Management.Automation.Host.ChoiceDescription]::new("&Yes", "Grant admin consent")
        [System.Management.Automation.Host.ChoiceDescription]::new("&No", "Skip admin consent")
    )
    
    $decision = $Host.UI.PromptForChoice($title, $question, $choices, 0)
    
    if ($decision -eq 0) {
        Write-StatusMessage "Granting admin consent..." -Color Green
        Start-Sleep -Seconds 20 # Allow time for app registration to propagate
        
        # Get the application
        $app = Get-MgApplication -ApplicationId $ApplicationId
        
        # Ensure service principal exists
        $servicePrincipal = Get-MgServicePrincipal -Filter "appId eq '$($app.AppId)'"
        if (-not $servicePrincipal) {
            $servicePrincipal = New-MgServicePrincipal -AppId $app.AppId
            Start-Sleep -Seconds 10
        }

        $hasError = $false
        foreach ($resourceAccess in $app.RequiredResourceAccess) {
            $resource = Get-MgServicePrincipal -Filter "appId eq '$($resourceAccess.ResourceAppId)'"
            
            if ($resource) {
                # Handle Application Permissions (AppRoles)
                $appRoles = $resourceAccess.ResourceAccess | Where-Object { $_.Type -eq "Role" }
                foreach ($permission in $appRoles) {
                    try {
                        $appRole = $resource.AppRoles | Where-Object { $_.Id -eq $permission.Id }
                        if ($appRole) {
                            $params = @{
                                PrincipalId = $servicePrincipal.Id
                                ResourceId = $resource.Id
                                AppRoleId = $permission.Id
                            }
                            
                            New-MgServicePrincipalAppRoleAssignment -ServicePrincipalId $servicePrincipal.Id -BodyParameter $params | Out-Null
                        }
                    }
                    catch {
                        if (-not ($_.Exception.Message -like "*Permission entry already exists*")) {
                            $hasError = $true
                            Write-StatusMessage "Failed to grant application permission: $_" -Color Red
                        }
                    }
                }

                # Handle Delegated Permissions (OAuth2PermissionScopes)
                $delegatedPermissions = $resourceAccess.ResourceAccess | Where-Object { $_.Type -eq "Scope" }
                if ($delegatedPermissions) {
                    try {
                        $scopes = @()
                        foreach ($permission in $delegatedPermissions) {
                            $scope = $resource.OAuth2PermissionScopes | Where-Object { $_.Id -eq $permission.Id }
                            if ($scope) {
                                $scopes += $scope.Value
                            }
                        }

                        if ($scopes.Count -gt 0) {
                            $params = @{
                                ClientId = $servicePrincipal.Id
                                ConsentType = "AllPrincipals"
                                ResourceId = $resource.Id
                                Scope = $scopes -join ' '
                            }
                            
                            $existingGrant = Get-MgOauth2PermissionGrant -Filter "clientId eq '$($servicePrincipal.Id)' and resourceId eq '$($resource.Id)'"
                            if ($existingGrant) {
                                $updatedScopes = ($existingGrant.Scope -split ' ' | Select-Object -Unique) + ($scopes | Select-Object -Unique)
                                $uniqueScopes = $updatedScopes | Select-Object -Unique
                                Update-MgOauth2PermissionGrant -OAuth2PermissionGrantId $existingGrant.Id -BodyParameter @{
                                    Scope = $uniqueScopes -join ' '
                                } | Out-Null
                            } else {
                                New-MgOauth2PermissionGrant -BodyParameter $params | Out-Null
                            }
                        }
                    }
                    catch {
                        $hasError = $true
                        Write-StatusMessage "Failed to grant delegated permissions: $_" -Color Red
                    }
                }
            }
        }
        
        if (-not $hasError) {
            Write-StatusMessage "Admin consent granted successfully." -Color Green
        }
    }
    else {
        Write-StatusMessage "Admin consent skipped." -Color Yellow
    }
}

# Main Script
try {
    Write-StatusMessage "Deployment started..."
    
    # Initialize environment
    Import-RequiredModules
    
    # Connect to Microsoft Graph with required permissions
    try {
        Write-StatusMessage "Connecting to Microsoft Graph..." -Color Cyan
        Disconnect-MgGraph -ErrorAction SilentlyContinue | Out-Null
        Connect-MgGraph -Scopes @(
            "Application.ReadWrite.All",
            "Directory.Read.All",
            "Directory.AccessAsUser.All"
        ) -NoWelcome -UseDeviceCode

        $context = Get-MgContext
        Write-StatusMessage "Connected to Microsoft Graph" -Color Green
        Write-StatusMessage "Account: $($context.Account)" -Color Cyan
        Write-StatusMessage "Environment: $($context.Environment)" -Color Cyan
        Write-StatusMessage "Tenant ID: $($context.TenantId)" -Color Cyan
        Write-StatusMessage "Scopes: $($context.Scopes -join ', ')" -Color Cyan

        if (-not (Get-MgContext)) {
            throw "Failed to establish Microsoft Graph connection"
        }
    } catch {
        Write-StatusMessage "Failed to connect to Microsoft Graph: $_" -Color Red
        throw
    }
    
    # Get input parameters
    $displayName = Read-Host -Prompt "Enter the Microsoft Entra ID app name (ex: Moodle plugin)"
    $moodleUrl = Get-MoodleUrl
    
    Write-Progress -Activity "Configuring application" -Status "Creating application" -PercentComplete 0
    
    # Setup URLs
    $replyUrls = @(
        $moodleUrl,
        "${moodleUrl}auth/oidc/",
        "${moodleUrl}local/o365/sso_end.php"
    )
    
    # Create application
    $requiredResourceAccess = Get-Resources
    
    $appParams = @{
        DisplayName = $displayName
        RequiredResourceAccess = $requiredResourceAccess
        Web = @{
            RedirectUris = $replyUrls
            HomePageUrl = $moodleUrl
            LogoutUrl = "${moodleUrl}auth/oidc/logout.php"
        }
    }
    
    $app = Invoke-WithRetry -OperationName "Application creation" {
        # Verify connection is valid
        $context = Get-MgContext
        if (-not $context -or -not $context.Account) {
            throw "Microsoft Graph connection is not available. Please ensure you are connected."
        }
        
        # Test connection with error handling
        try {
            # Use a simple test that doesn't require additional permissions
            $null = Get-MgContext
            if (-not (Get-MgContext).Account) {
                throw "Authentication context is invalid"
            }
        }
        catch {
            Write-StatusMessage "Connection test failed, attempting to reconnect..." -Color Yellow
            
            # Clean disconnect and reconnect
            try {
                Disconnect-MgGraph -ErrorAction SilentlyContinue | Out-Null
                Start-Sleep -Seconds 2
                
                Connect-MgGraph -Scopes @(
                    "Application.ReadWrite.All",
                    "Directory.Read.All", 
                    "Directory.AccessAsUser.All"
                ) -NoWelcome -UseDeviceCode -ErrorAction Stop
                
                Start-Sleep -Seconds 3
                
                if (-not (Get-MgContext).Account) {
                    throw "Failed to establish valid connection"
                }
            }
            catch {
                throw "Unable to establish Microsoft Graph connection: $_"
            }
        }
        
        # Create application with basic parameters first
        $basicParams = @{
            DisplayName = $appParams.DisplayName
            RequiredResourceAccess = $appParams.RequiredResourceAccess
        }
        
        Write-StatusMessage "Creating application with display name: $($basicParams.DisplayName)" -Color Cyan
        
        try {
            $newApp = New-MgApplication @basicParams
            
            if (-not $newApp -or -not $newApp.Id) {
                throw "Application creation failed - no application object returned"
            }
            
            # Wait a moment before updating
            Start-Sleep -Seconds 2
            
            # Update with web configuration separately
            Write-StatusMessage "Updating application with web configuration..." -Color Cyan
            Update-MgApplication -ApplicationId $newApp.Id -Web $appParams.Web
            
            return $newApp
        }
        catch {
            if ($_.Exception.Message -like "*DeviceCodeCredential*") {
                throw "Authentication failed. Please ensure you complete the device code authentication process."
            }
            throw
        }
    }

    # Add the new Enterprise Application code here, before the API access configuration
    Write-Progress -Activity "Configuring application" -Status "Creating Enterprise Application" -PercentComplete 20

    $servicePrincipal = Invoke-WithRetry -OperationName "Enterprise Application creation" {
        # Check if service principal already exists
        $existingSP = Get-MgServicePrincipal -Filter "appId eq '$($app.AppId)'"
        
        if (-not $existingSP) {
            Write-StatusMessage "Creating Enterprise Application..." -Color Cyan
            New-MgServicePrincipal -AppId $app.AppId -Tags @("WindowsAzureActiveDirectoryIntegratedApp")
        } else {
            Write-StatusMessage "Enterprise Application already exists" -Color Cyan
            $existingSP
        }
    }

    Write-StatusMessage "Enterprise Application ID: $($servicePrincipal.Id)"
    
    Write-StatusMessage "Successfully registered app with ID: $($app.AppId)"
    
    # Configure API access
    Write-Progress -Activity "Configuring application" -Status "Configuring API access" -PercentComplete 30
    
    # Get verified domains first
    Write-StatusMessage "Checking verified domains..." -Color Cyan
    $verifiedDomains = @()
    try {
        $domains = Get-MgDomain
        $verifiedDomains = $domains | Where-Object { $_.IsVerified -eq $true } | ForEach-Object { $_.Id }
        Write-StatusMessage "Found verified domains: $($verifiedDomains -join ', ')" -Color Cyan
    }
    catch {
        Write-StatusMessage "Warning: Could not retrieve verified domains: $_" -Color Yellow
    }

    # Check if Moodle URL contains a verified domain
    $moodleDomain = ([System.Uri]$moodleUrl).Host
    $useCustomUri = $false
    
    if ($verifiedDomains.Count -gt 0) {
        foreach ($domain in $verifiedDomains) {
            if ($moodleDomain -like "*$domain*" -or $moodleDomain -eq $domain) {
                $useCustomUri = $true
                Write-StatusMessage "Moodle domain '$moodleDomain' matches verified domain '$domain'" -Color Green
                break
            }
        }
    }

    if ($useCustomUri) {
        $identifierUris = "api://$($moodleUrl.Replace('https://', ''))$($app.AppId)"
        Write-StatusMessage "Using custom identifier URI: $identifierUris" -Color Cyan
    } else {
        $identifierUris = "api://$($app.AppId)"
        Write-StatusMessage "Using fallback identifier URI (domain not verified): $identifierUris" -Color Yellow
    }
    
    $apiParams = @{
        ApplicationId = $app.Id
        IdentifierUris = @($identifierUris)
        Api = @{
            Oauth2PermissionScopes = @(
                @{
                    AdminConsentDescription = "Allows Teams to call the app's web APIs as the current user"
                    AdminConsentDisplayName = "Teams can access the user's profile"
                    UserConsentDescription = "Enable Teams to call this app's APIs with the same rights as the user"
                    UserConsentDisplayName = "Teams can access the user profile and make requests on the user's behalf"
                    IsEnabled = $true
                    Type = "User"
                    Value = "access_as_user"
                    Id = [Guid]::NewGuid()
                }
            )
        }
    }

    Invoke-WithRetry -OperationName "API configuration" {
        Update-MgApplication @apiParams
    }
    Write-StatusMessage "API configuration completed successfully" -Color Green
    
    # Add current user as owner
    Write-Progress -Activity "Configuring application" -Status "Adding owner" -PercentComplete 60
    
    $context = Get-MgContext
    $user = Get-MgUser -UserId $context.Account
    
    Invoke-WithRetry -OperationName "Owner addition" {
        New-MgApplicationOwnerByRef -ApplicationId $app.Id -OdataId "https://graph.microsoft.com/v1.0/directoryObjects/$($user.Id)"
    }
    
    # Generate client secret
    Write-Progress -Activity "Configuring application" -Status "Generating credentials" -PercentComplete 80
    
    $secret = Invoke-WithRetry -OperationName "Secret creation" {
        Add-MgApplicationPassword -ApplicationId $app.Id
    }
    
    # Update logo
    $logoPath = Join-Path (Get-Location) 'Assets\moodle-logo.jpg'
    if (Test-Path $logoPath) {
        Invoke-WithRetry -OperationName "Logo update" {
            Set-MgApplicationLogo -ApplicationId $app.Id -InFile $logoPath
        }
    }
    
    # Grant admin consent if requested
    Grant-AdminConsent -ApplicationId $app.Id
    
    # Configure optional claims
    $optionalClaimsPath = Join-Path (Get-Location) 'Json\EntraIDOptionalClaims.json'
    if (Test-Path $optionalClaimsPath) {
        try {
            $optionalClaimsJson = Get-Content $optionalClaimsPath | ConvertFrom-Json
            
            # Create the OptionalClaims object
            $optionalClaimsObj = [Microsoft.Graph.PowerShell.Models.MicrosoftGraphOptionalClaims]::new()
            
            # Process ID Token claims
            if ($optionalClaimsJson.idToken) {
                $optionalClaimsObj.IdToken = @()
                foreach ($claim in $optionalClaimsJson.idToken) {
                    $optionalClaim = [Microsoft.Graph.PowerShell.Models.MicrosoftGraphOptionalClaim]::new()
                    $optionalClaim.Name = $claim.name
                    $optionalClaim.Source = $claim.source
                    $optionalClaim.Essential = $claim.essential
                    $optionalClaim.AdditionalProperties = $claim.additionalProperties
                    $optionalClaimsObj.IdToken += $optionalClaim
                }
            }

            # Process Access Token claims
            if ($optionalClaimsJson.accessToken) {
                $optionalClaimsObj.AccessToken = @()
                foreach ($claim in $optionalClaimsJson.accessToken) {
                    $optionalClaim = [Microsoft.Graph.PowerShell.Models.MicrosoftGraphOptionalClaim]::new()
                    $optionalClaim.Name = $claim.name
                    $optionalClaim.Source = $claim.source
                    $optionalClaim.Essential = $claim.essential
                    $optionalClaim.AdditionalProperties = $claim.additionalProperties
                    $optionalClaimsObj.AccessToken += $optionalClaim
                }
            }

            # Process SAML2 Token claims
            if ($optionalClaimsJson.saml2Token) {
                $optionalClaimsObj.Saml2Token = @()
                foreach ($claim in $optionalClaimsJson.saml2Token) {
                    $optionalClaim = [Microsoft.Graph.PowerShell.Models.MicrosoftGraphOptionalClaim]::new()
                    $optionalClaim.Name = $claim.name
                    $optionalClaim.Source = $claim.source
                    $optionalClaim.Essential = $claim.essential
                    $optionalClaim.AdditionalProperties = $claim.additionalProperties
                    $optionalClaimsObj.Saml2Token += $optionalClaim
                }
            }

            Update-MgApplication -ApplicationId $app.Id -OptionalClaims $optionalClaimsObj
            Write-StatusMessage "Microsoft Entra ID optional claims updated" -Color Green
        }
        catch {
            Write-StatusMessage "Failed to add Mcrosoft Entra ID optional claims: $_" -Color Red
        }
    }
    
    # Configure Teams integration
    Write-Progress -Activity "Configuring application" -Status "Configuring Teams integration" -PercentComplete 90
    
    # Get the access_as_user scope
    $scope = Invoke-WithRetry -OperationName "Get scope" {
        $updatedApp = Get-MgApplication -ApplicationId $app.Id
        $updatedApp.Api.Oauth2PermissionScopes | Where-Object { $_.Value -eq 'access_as_user' }
    }
    
    if ($scope) {
        $preAuthorizedApps = @(
            @{
                AppId = '1fec8e78-bce4-4aaf-ab1b-5451cc387264'  # Teams Rich Client
                DelegatedPermissionIds = @($scope.Id)
            },
            @{
                AppId = '5e3ce6c0-2b1f-4285-8d4b-75ee78787346'  # Teams Web Client
                DelegatedPermissionIds = @($scope.Id)
            }
        )
        
        Invoke-WithRetry -OperationName "Teams pre-authorization" {
            Update-MgApplication -ApplicationId $app.Id -Api @{
                PreAuthorizedApplications = $preAuthorizedApps
            }
        }
    }
    
    # Output results
    Write-Progress -Activity "Configuring application" -Completed
    Write-StatusMessage "`nApplication configuration completed successfully!"
    Write-StatusMessage "Application (Client) ID: $($app.AppId)"
    Write-StatusMessage "Client Secret: $($secret.SecretText)"
    Write-StatusMessage "Please save these credentials securely."
}
catch {
    Write-StatusMessage "Deployment failed: $_" -Color Red
    exit 1
}