<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Text (on cohorts) - Edit form
 *
 * @package   block_cohortspecifichtml
 * @copyright 2017 Kathrin Osswald, Ulm University kathrin.osswald@uni-ulm.de
 *            based on code from 2010 Petr Skoda (http://skodak.org)
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * Form for editing Text block instances.
 *
 * @param stdClass $course course object
 * @param stdClass $birecordorcm block instance record
 * @param context $context context object
 * @param string $filearea file area
 * @param array $args extra arguments
 * @param bool $forcedownload whether or not force download
 * @param array $options additional options affecting the file serving
 * @return bool
 * @todo MDL-36050 improve capability check on stick blocks, so we can check user capability before sending images.
 */
function block_cohortspecifichtml_pluginfile(
    $course,
    $birecordorcm,
    $context,
    $filearea,
    $args,
    $forcedownload,
    array $options = []
) {
    global $CFG;
    require_once("{$CFG->dirroot}/user/lib.php");
    require_once($CFG->dirroot . '/blocks/cohortspecifichtml/locallib.php');

    if ($context->contextlevel != CONTEXT_BLOCK) {
        send_file_not_found();
    }

    // If block is in course context, then check if user has capability to access course.
    if ($context->get_course_context(false)) {
        require_course_login($course);
    } else if ($CFG->forcelogin) {
        require_login();
    } else {
        // Get parent context and see if user have proper permission.
        $parentcontext = $context->get_parent_context();
        if ($parentcontext->contextlevel === CONTEXT_COURSECAT) {
            // Check if category is visible and user can view this category.
            if (!core_course_category::get($parentcontext->instanceid, IGNORE_MISSING)) {
                send_file_not_found();
            }
        } else if ($parentcontext->contextlevel === CONTEXT_USER) {
            $user = core_user::get_user($parentcontext->instanceid, '*', MUST_EXIST);
            $extracaps = block_method_result('cohortspecifichtml', 'get_extra_capabilities');
            if (!user_can_view_profile($user, null, $parentcontext) || !has_any_capability($extracaps, $context)) {
                send_file_not_found();
            }
        }
        // At this point there is no way to check SYSTEM context, so ignoring it.
    }

    if ($filearea !== 'content') {
        send_file_not_found();
    }

    $fs = get_file_storage();

    $filename = array_pop($args);
    $filepath = $args ? '/' . implode('/', $args) . '/' : '/';

    // The following three lines diverge from the code in block_html which served as blueprint for
    // this function just because of the fact that block_html used 'and' operators which
    // are now discouraged by the Moodle codechecker.
    $file = $fs->get_file($context->id, 'block_cohortspecifichtml', 'content', 0, $filepath, $filename);
    if (($file === false) || ($file->is_directory() === true)) {
        send_file_not_found();
    }

    // Check the same cohort conditions for the file. If not valid, do send_file_not_found.
    $instance = block_instance($birecordorcm->blockname, $birecordorcm);
    if (
        block_cohortspecifichtml_show_block($instance) != true &&
            block_cohortspecifichtml_get_caneditandediton($instance) != true
    ) {
        send_file_not_found();
    }

    if ($parentcontext = context::instance_by_id($birecordorcm->parentcontextid, IGNORE_MISSING)) {
        if ($parentcontext->contextlevel == CONTEXT_USER) {
            // Force download on all personal pages including /my/ ...
            // ... because we do not have reliable way to find out from where this is used.
            $forcedownload = true;
        }
    } else {
        // Weird, there should be parent context, better force dowload then.
        $forcedownload = true;
    }

    // NOTE: it woudl be nice to have file revisions here, for now rely on standard file lifetime ...
    // ... do not lower it because the files are dispalyed very often.
    \core\session\manager::write_close();
    send_stored_file($file, null, 0, $forcedownload, $options);
}

/**
 * Perform global search replace such as when migrating site to new URL.
 * @param  string $search
 * @param  string $replace
 * @return void
 */
function block_cohortspecifichtml_global_db_replace($search, $replace) {
    global $DB;

    $instances = $DB->get_recordset('block_instances', ['blockname' => 'cohortspecifichtml']);
    foreach ($instances as $instance) {
        // In block_html, it says: Intentionally hardcoded until MDL-26800 is fixed.
        $config = unserialize_object(base64_decode($instance->configdata));
        if (isset($config->text) && is_string($config->text)) {
            $config->text = str_replace($search, $replace, $config->text);
            $DB->set_field('block_instances', 'configdata', base64_encode(serialize($config)), ['id' => $instance->id]);
        }
    }
    $instances->close();
}

/**
 * Given an array with a file path, it returns the itemid and the filepath for the defined filearea.
 *
 * @param  string $filearea The filearea.
 * @param  array  $args The path (the part after the filearea and before the filename).
 * @return array The itemid and the filepath inside the $args path, for the defined filearea.
 */
function block_cohortspecifichtml_get_path_from_pluginfile(string $filearea, array $args): array {
    // This block never has an itemid (the number represents the revision but it's not stored in database).
    array_shift($args);

    // Get the filepath.
    if (empty($args)) {
        $filepath = '/';
    } else {
        $filepath = '/' . implode('/', $args) . '/';
    }

    return [
            'itemid' => 0,
            'filepath' => $filepath,
    ];
}
