<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Codeset form.
 *
 * @package    enrol_accesscodes
 * @copyright  2025 Andy McGill {@link https://brookesia.co.uk}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace enrol_accesscodes;

use moodleform;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once("$CFG->libdir/formslib.php");

/**
 * Form to create/edit codesets.
 */
class codeset_form extends moodleform {
    /**
     * Form definition.
     * @throws moodle_exception
     */
    public function definition() {
        global $DB;

        $mform = $this->_form;
        $customdata = $this->_customdata;

        $mform->addElement('hidden', 'id');
        $mform->setType('id', PARAM_INT);

        // Name.
        $nameattributes = ['placeholder' => get_string('codesetname', 'enrol_accesscodes')];
        $mform->addElement('text', 'codeset_name', get_string('codesetname', 'enrol_accesscodes'), $nameattributes);
        $mform->setType('codeset_name', PARAM_TEXT);
        $mform->addRule('codeset_name', get_string('required'), 'required');

        // All courses.
        $mform->addElement('selectyesno', 'all_courses', get_string('allcourses', 'enrol_accesscodes'));
        $mform->setType('all_courses', PARAM_INT);
        $mform->addHelpButton('all_courses', 'allcourses', 'enrol_accesscodes');

        // Valid courses.
        $sql = "SELECT c.id, c.fullname
                FROM {course} c
                JOIN {enrol} e ON e.courseid = c.id
                WHERE e.enrol = 'accesscodes'
                ";
        $coursesavailable = $DB->get_records_sql($sql);

        $coursesavailablemapped = array_column($coursesavailable, 'fullname', 'id');

        $validcoursesoptions = [
            'multiple' => true,
        ];
        $mform->addElement(
            'autocomplete',
            'valid_courses',
            get_string('validcourses', 'enrol_accesscodes'),
            $coursesavailablemapped,
            $validcoursesoptions
        );
        $mform->setType('valid_courses', PARAM_INT);
        $mform->disabledIf('valid_courses', 'all_courses', 'eq', 1);
        $mform->hideIf('valid_courses', 'all_courses', 'eq', 1);
        $mform->addHelpButton('valid_courses', 'validcourses', 'enrol_accesscodes');

        // Time expires.
        $timeexpiresoptions = [
            'startyear' => date("Y"),
            'optional' => true,
        ];
        $mform->addElement(
            'date_time_selector',
            'time_expires',
            get_string('expirydate', 'enrol_accesscodes'),
            $timeexpiresoptions
        );
        $mform->addHelpButton('time_expires', 'expirydate', 'enrol_accesscodes');

        // Enrol period.
        $enrolperiodoptions = [
            'optional' => true,
        ];
        $mform->addElement(
            'duration',
            'enrol_period',
            get_string('enrolmentduration', 'enrol_accesscodes'),
            $enrolperiodoptions
        );
        $mform->addHelpButton('enrol_period', 'enrolmentduration', 'enrol_accesscodes');

        // Number of codes to generate.
        if (!empty($customdata['existing_id'])) {
            $additionalcodesattributes = ['placeholder' => get_string('additionalcodes', 'enrol_accesscodes')];
            $mform->addElement(
                'float',
                'additional_codes',
                get_string('additionalcodes', 'enrol_accesscodes'),
                $additionalcodesattributes
            );
            $mform->setType('additional_codes', PARAM_INT);
            $this->add_action_buttons();
        } else {
            $mform->addElement(
                'html',
                '<p class="alert alert-danger">' . get_string('codescantbereduced', 'enrol_accesscodes') . '</p>'
            );

            $numberofcodesattributes = ['placeholder' => get_string('numberofcodes', 'enrol_accesscodes')];
            $mform->addElement(
                'float',
                'number_of_codes',
                get_string('numberofcodes', 'enrol_accesscodes'),
                $numberofcodesattributes
            );
            $mform->setType('number_of_codes', PARAM_INT);
            $mform->addRule('number_of_codes', get_string('nonzerocodeserror', 'enrol_accesscodes'), 'nonzero');

            $this->add_action_buttons(true, get_string('saveandgenerate', 'enrol_accesscodes'));
        }
    }

    /**
     * Form validation.
     * @param array $data
     * @param array $files
     * @return array
     */
    public function validation($data, $files) {
        $errors = [];

        // If all_courses == 0 at least 1 valid_courses id should be set.
        if ($data['all_courses'] == 0 && empty($data['valid_courses'])) {
            $errors['valid_courses'] = get_string('coursesrequirederror', 'enrol_accesscodes');
        }

        return $errors;
    }
}
