<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace enrol_accesscodes\external;

use core_external\external_function_parameters;
use core_external\external_multiple_structure;
use core_external\external_single_structure;
use core_external\external_value;
use core_external\external_warnings;
use context_course;
use context_system;
use core_course_category;
use moodle_exception;

/**
 * Access codes enrolment external functions.
 *
 * @package    enrol_accesscodes
 * @copyright  2025 Andy McGill {@link https://brookesia.co.uk}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class enrol_user extends \core_external\external_api {
    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function execute_parameters() {
        return new external_function_parameters(
            [
                'courseid' => new external_value(PARAM_INT, 'Id of the course'),
                'code' => new external_value(PARAM_RAW, 'Access code'),
                'instanceid' => new external_value(PARAM_INT, 'Instance id of accesscodes enrolment plugin.', VALUE_DEFAULT, 0),
            ]
        );
    }

    /**
     * Access code enrol the current user in the given course.
     *
     * @param int $courseid id of course
     * @param string $code access code
     * @param int $instanceid instance id of access code enrolment plugin
     * @return array of warnings and status result
     * @since Moodle 3.0
     * @throws moodle_exception
     */
    public static function execute($courseid, $code, $instanceid = 0) {
        global $CFG;

        require_once($CFG->libdir . '/enrollib.php');

        $params = self::validate_parameters(
            self::execute_parameters(),
            [
                'courseid' => $courseid,
                'code' => $code,
                'instanceid' => $instanceid,
            ]
        );

        $warnings = [];

        $course = get_course($params['courseid']);
        self::validate_context(context_system::instance());

        if (!core_course_category::can_view_course_info($course)) {
            throw new moodle_exception('coursehidden');
        }

        // Retrieve the access codes enrolment plugin.
        $enrol = enrol_get_plugin('accesscodes');
        if (empty($enrol)) {
            throw new moodle_exception('cannotenrol', 'enrol_accesscodes');
        }

        // We can expect multiple access code enrolment instances.
        $instances = [];
        $enrolinstances = enrol_get_instances($course->id, true);
        foreach ($enrolinstances as $courseenrolinstance) {
            if ($courseenrolinstance->enrol == "accesscodes") {
                // Instance specified.
                if (!empty($params['instanceid'])) {
                    if ($courseenrolinstance->id == $params['instanceid']) {
                        $instances[] = $courseenrolinstance;
                        break;
                    }
                } else {
                    $instances[] = $courseenrolinstance;
                }
            }
        }
        if (empty($instances)) {
            throw new moodle_exception('cannotenrol', 'enrol_accesscodes');
        }

        // Try to enrol the user in the instance/s.
        $enrolled = false;
        foreach ($instances as $instance) {
            $enrolstatus = $enrol->can_self_enrol($instance);
            if ($enrolstatus === true) {
                require_once($CFG->dirroot . "/enrol/accesscodes/locallib.php");

                $accesscodevalid = enrol_accesscodes_check_access_code($course->id, $params['code']);

                if (true !== $accesscodevalid) {
                    $warnings[] = [
                        'item' => 'instance',
                        'itemid' => $instance->id,
                        'warningcode' => '4',
                        'message' => $accesscodevalid,
                    ];
                    continue;
                }

                // Do the enrolment.
                $data = ['accesscode' => $params['code']];
                $enrol->enrol_accesscodes($instance, (object) $data);
                $enrolled = $accesscodevalid;
                break;
            } else {
                $warnings[] = [
                    'item' => 'instance',
                    'itemid' => $instance->id,
                    'warningcode' => '1',
                    'message' => $enrolstatus,
                ];
            }
        }

        $result = [];
        $result['status'] = $enrolled;
        $result['warnings'] = $warnings;
        return $result;
    }

    /**
     * Returns description of method result value
     *
     * @return external_description
     * @since Moodle 3.0
     */
    public static function execute_returns() {
        return new external_single_structure(
            [
                'status' => new external_value(PARAM_BOOL, 'status: true if the user is enrolled, false otherwise'),
                'warnings' => new external_warnings(),
            ]
        );
    }
}
