<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 *
 * @package    enrol_accesscodes
 * @copyright  2025 Andy McGill {@link https://brookesia.co.uk}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require('../../config.php');
require_once($CFG->libdir . '/adminlib.php');
require_once($CFG->dirroot . '/enrol/accesscodes/classes/codeset_form.php');

admin_externalpage_setup('enrol_accesscodes/managecodesets');

$systemcontext = context_system::instance();

$hasaccessto = has_capability('enrol/accesscodes:manageallcodes', $systemcontext) ? 'all' : 'own';

$codesetid = required_param('id', PARAM_INT);
$codeset = $DB->get_record('enrol_accesscodes_codesets', ['id' => $codesetid]);

if ($codeset->creatorid != $USER->id && $hasaccessto != 'all') {
    throw new \moodle_exception('nopermissions', 'error', '', get_string('editexception', 'enrol_accesscodes'));
}

$thisurl = new moodle_url('/enrol/accesscodes/edit_codeset.php', ['id' => $codesetid]);

$heading = get_string('edit') . ' ' . $codeset->name;

$currentcourses = $DB->get_records('enrol_accesscodes_cs_crs', ['codesetid' => $codeset->id], '', 'courseid');
$currentcourseids = array_column($currentcourses, 'courseid');

$PAGE->set_title($heading);
$PAGE->set_heading($heading);
$PAGE->set_url($thisurl);
$PAGE->set_context($systemcontext);

$breadcrumbrootstring = $hasaccessto == 'all' ?
    get_string('managecodes', 'enrol_accesscodes') : get_string('managemycodes', 'enrol_accesscodes');

$PAGE->navbar->ignore_active();
$PAGE->navbar->add(
    $breadcrumbrootstring,
    new moodle_url('/enrol/accesscodes/manage_codesets.php')
);

$itemnode = $PAGE->navbar->add($heading);

$customformdata = ['existing_id' => $codesetid];
$mform = new enrol_accesscodes\codeset_form(null, $customformdata);

// Form processing and displaying is done here.
if ($mform->is_cancelled()) {
    $returnurl = new moodle_url('/enrol/accesscodes/manage_codesets.php');
    redirect($returnurl);
} else if ($formdata = $mform->get_data()) {
    $codeset->name = $formdata->codeset_name;
    $codeset->allcourses = $formdata->all_courses;
    $codeset->timeexpires = $formdata->time_expires;
    $codeset->enrolperiod = $formdata->enrol_period;
    $codeset->timemodified = time();

    $DB->update_record('enrol_accesscodes_codesets', $codeset);

    if ($formdata->all_courses == 0) {
        // Update courses.

        // Remove any that aren't there.
        $removedcourses = array_diff($currentcourseids, $formdata->valid_courses);
        foreach ($removedcourses as $del) {
            $DB->delete_records("enrol_accesscodes_cs_crs", ['courseid' => $del, 'codesetid' => $codeset->id]);
        }

        // Add any new ones.
        foreach ($formdata->valid_courses as $courseid) {
            if (!in_array($courseid, $currentcourseids)) {
                $cscrsrecord = (object) [
                    'codesetid' => $codeset->id,
                    'courseid' => $courseid,
                ];
                $DB->insert_record('enrol_accesscodes_cs_crs', $cscrsrecord);
            }
        }
    }

    if (!empty($formdata->additional_codes)) {
        require_once($CFG->dirroot . '/enrol/accesscodes/classes/accesscodes_helper.php');
        $newcodes = enrol_accesscodes\accesscodes_helper::generate_codes($formdata->additional_codes);
        foreach ($newcodes as $c) {
            $coderecord = (object) [
                'code' => $c,
                'codesetid' => $codeset->id,
                'status' => 'available',
                'timecreated' => time(),
                'timemodified' => time(),
            ];
            $DB->insert_record('enrol_accesscodes_codes', $coderecord);
        }
    }

    \core\notification::add(get_string('codesetupdated', 'enrol_accesscodes'), \core\notification::SUCCESS);
    $returnurl = new moodle_url('/enrol/accesscodes/manage_codesets.php');
    redirect($returnurl);
} else {
    // This branch is executed if the form is submitted but the data doesn't.
    // Validate and the form should be redisplayed or on the first display of the form.

    $existingdata = [
        'id' => $codeset->id,
        'codeset_name' => $codeset->name,
        'all_courses' => $codeset->allcourses,
        'time_expires' => $codeset->timeexpires,
        'enrol_period' => $codeset->enrolperiod,
        'valid_courses' => $currentcourseids,
    ];
    // Set default data.
    $mform->set_data($existingdata);
}

echo $OUTPUT->header();

$mform->display();

echo $OUTPUT->footer();
