<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 *
 * @package    enrol_accesscodes
 * @copyright  2025 Andy McGill {@link https://brookesia.co.uk}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require('../../config.php');
require_once($CFG->libdir . '/adminlib.php');
require_once('./classes/enrol_accesscodes_configduration.php');

admin_externalpage_setup('enrol_accesscodes/managecodesets');

$systemcontext = context_system::instance();

$hasaccessto = has_capability('enrol/accesscodes:manageallcodes', $systemcontext) ? 'all' : 'own';

$codesetid = required_param('id', PARAM_INT);
$codeset = $DB->get_record('enrol_accesscodes_codesets', ['id' => $codesetid]);

if ($codeset->creatorid != $USER->id && $hasaccessto != 'all') {
    throw new \moodle_exception('nopermissions', 'error', '', get_string('viewexception', 'enrol_accesscodes'));
}

$csv = optional_param('csv', 0, PARAM_INT);

$thisurl = new moodle_url('/enrol/accesscodes/view_codes.php', ['id' => $codesetid]);

$heading = get_string('accesscodes', 'enrol_accesscodes');

$sql = "SELECT c.fullname from {course} c
        JOIN {enrol_accesscodes_cs_crs} cs_crs ON cs_crs.courseid = c.id
        WHERE cs_crs.codesetid = :codesetid";
$courses = $DB->get_records_sql($sql, ['codesetid' => $codesetid]);

$translatedused = get_string('used', 'enrol_accesscodes');
$translatedavailable = get_string('available', 'enrol_accesscodes');

$sql = "SELECT c.*,
        CASE WHEN c.status = 'used' THEN 1 ELSE 0 END AS used,
        CASE WHEN c.status = 'used' THEN '{$translatedused}' ELSE '{$translatedavailable}' END AS status_translated,
        -- IF(c.status = 'used', 1, 0) as used,
        -- IF(c.status = 'used', '{$translatedused}', '{$translatedavailable}') as status_translated,
        crs.fullname as course_name, CONCAT(u.firstname,' ',u.lastname) as user_name
        FROM {enrol_accesscodes_codes} c
        LEFT JOIN {course} crs ON crs.id = c.courseid
        LEFT JOIN {user} u ON u.id = c.userid
        WHERE codesetid = :codesetid
        ORDER BY c.id";

$codes = $DB->get_records_sql($sql, ['codesetid' => $codesetid]);

$codesunused = $DB->count_records('enrol_accesscodes_codes', ['codesetid' => $codesetid, 'status' => 'available']);

$expirydate = !empty($codeset->timeexpires) ? date('d/m/Y H:i:s', $codeset->timeexpires) : get_string('never');

$enrolperiod = !empty($codeset->enrolperiod) ?
    enrol_accesscodes_configduration::get_duration_as_text($codeset->enrolperiod)
    : get_string('default');

if ($csv) {
    require_sesskey();
    require_once($CFG->libdir . '/csvlib.class.php');
    $writer = new csv_export_writer();
    $filename = clean_param($codeset->name . '-codes', PARAM_FILE);
    $writer->set_filename($filename);

    $headers = [
        get_string('code', 'enrol_accesscodes'),
        get_string('status'),
        get_string('usedfor', 'enrol_accesscodes'),
    ];

    $writer->add_data($headers);

    foreach ($codes as $code) {
        $row = [
            $code->code,
            $code->status,
            $code->course_name ?? "-",
        ];
        $writer->add_data($row);
    }

    $writer->download_file();
    exit();
}

$pagedata = [
    'codeset' => $codeset,
    'expiry_date' => $expirydate,
    'enrol_period' => $enrolperiod,
    'courses_string' => implode(", ", array_column($courses, 'fullname')),
    'courses' => array_values($courses),
    'codes' => array_values($codes),
    'number_of_codes' => count($codes),
    'codes_unused' => $codesunused,
    'plugin_url' => $CFG->wwwroot . '/enrol/accesscodes',
    'export_url' => $CFG->wwwroot . '/enrol/accesscodes/view_codes.php?id=' . $codesetid . '&csv=1&sesskey=' . sesskey(),
];

$PAGE->set_title($codeset->name);
$PAGE->set_heading($codeset->name);
$PAGE->set_url($thisurl);
$PAGE->set_context($systemcontext);

$breadcrumbrootstring = $hasaccessto == 'all' ?
    get_string('managecodes', 'enrol_accesscodes') : get_string('managemycodes', 'enrol_accesscodes');

$PAGE->navbar->ignore_active();
$PAGE->navbar->add(
    $breadcrumbrootstring,
    new moodle_url('/enrol/accesscodes/manage_codesets.php')
);

$itemnode = $PAGE->navbar->add(
    $codeset->name
);

echo $OUTPUT->header();

echo $OUTPUT->render_from_template('enrol_accesscodes/view_codes', $pagedata);

echo $OUTPUT->footer();
