<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Access codes enrol plugin implementation.
 *
 * @package    enrol_accesscodes
 * @copyright  2025 Andy McGill {@link https://brookesia.co.uk}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

require_once("$CFG->libdir/formslib.php");
require_once($CFG->dirroot . '/enrol/locallib.php');


/**
 * Check if the given password match an access code in the specified course.
 *
 * @param  int $courseid            course id
 * @param  string $accesscode       access code
 * @return bool|string              True if valid, error message if not
 * @since  Moodle 3.0
 */
function enrol_accesscodes_check_access_code($courseid, $accesscode) {

    global $DB;

    if (
        $accesscode = $DB->get_record(
            'enrol_accesscodes_codes',
            ['code' => $accesscode, 'status' => 'available']
        )
    ) {
        $codeset = $DB->get_record('enrol_accesscodes_codesets', ['id' => $accesscode->codesetid]);

        // Check this code has not expired.
        $now = time();
        if ($now >= $codeset->timeexpires && !empty($codeset->timeexpires)) {
            return get_string("expiredcode", 'enrol_accesscodes');
        }
        // Check this code is allowed on this course.
        if (!$codeset->allcourses) {
            if (!$DB->record_exists('enrol_accesscodes_cs_crs', ['codesetid' => $codeset->id, 'courseid' => $courseid])) {
                return get_string("invalidforcourse", 'enrol_accesscodes');
            }
        }
        // If we get here it's valid.
        return true;
    } else {
        return get_string("invalidcode", 'enrol_accesscodes');
    }
}

/**
 * Enrol form implementation
 */
class enrol_accesscodes_enrol_form extends moodleform {
    /** @var object $instance */
    protected $instance;
    /** @var bool $toomany */
    protected $toomany = false;

    /**
     * Overriding this function to get unique form id for multiple self enrolments.
     *
     * @return string form identifier
     */
    protected function get_form_identifier() {
        $formid = $this->_customdata->id . '_' . get_class($this);
        return $formid;
    }

    /**
     * Form definition
     */
    public function definition() {
        $mform = $this->_form;
        $instance = $this->_customdata;
        $this->instance = $instance;
        $plugin = enrol_get_plugin('accesscodes');

        $heading = $plugin->get_instance_name($instance);
        $mform->addElement('html', '<h5>' . $heading . '</h5>');

        $mform->addElement(
            'password',
            'accesscode',
            get_string('accesscode', 'enrol_accesscodes'),
            ['id' => 'accesscode_' . $instance->id]
        );

        $this->add_action_buttons(false, get_string('enrolme', 'enrol_accesscodes'));

        $mform->addElement('hidden', 'id');
        $mform->setType('id', PARAM_INT);
        $mform->setDefault('id', $instance->courseid);

        $mform->addElement('hidden', 'instance');
        $mform->setType('instance', PARAM_INT);
        $mform->setDefault('instance', $instance->id);
    }

    /**
     * Form validation
     */
    public function validation($data, $files) {
        global $DB, $CFG, $USER;

        $errors = parent::validation($data, $files);
        $instance = $this->instance;

        if ($this->toomany) {
            $errors['notice'] = get_string('error');
            return $errors;
        }

        $accesscodevalid = enrol_accesscodes_check_access_code($instance->courseid, $data['accesscode']);

        if (true !== $accesscodevalid) {
            $errors['accesscode'] = $accesscodevalid;
        }

        return $errors;
    }
}
