<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 *
 * @package    enrol_accesscodes
 * @copyright  2025 Andy McGill {@link https://brookesia.co.uk}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require('../../config.php');
require_once($CFG->libdir . '/adminlib.php');

admin_externalpage_setup('enrol_accesscodes/managecodesets');

$systemcontext = context_system::instance();

$hasaccessto = has_capability('enrol/accesscodes:manageallcodes', $systemcontext) ? 'all' : 'own';

$search = optional_param('search', null, PARAM_TEXT);

$showall = optional_param('showall', null, PARAM_TEXT);

if ($showall == "on" && $hasaccessto == 'own') {
    $showall = null;
}

$page = optional_param('page', 0, PARAM_INT); // Needs to be zero indexed for the pagingbar.
$limit = 20;
$offset = $limit * $page;

if (!empty($search) || !empty($showall)) {
    $urlparams = [];
    if (!empty($search)) {
        $urlparams['search'] = $search;
    }
    if (!empty($showall)) {
        $urlparams['showall'] = $showall;
    }
    $thisurl = new moodle_url('/enrol/accesscodes/manage_codesets.php', $urlparams);
} else {
    $thisurl = new moodle_url('/enrol/accesscodes/manage_codesets.php');
}

$heading = has_capability('enrol/accesscodes:manageallcodes', $systemcontext) ?
    get_string('managecodes', 'enrol_accesscodes') : get_string('managemycodes', 'enrol_accesscodes');


$sql = "SELECT cs.*,
        CONCAT(u.firstname, ' ', u.lastname) AS created_by,
        cs.timecreated as date_created,
        -- TO_CHAR(TO_TIMESTAMP(cs.timecreated), 'DD/MM/YYYY') AS date_created, -- Only works in postgres
        -- DATE_FORMAT(FROM_UNIXTIME(cs.timecreated), '%d/%m/%Y') as date_created, -- Only works in MariDB/MySQL
        CONCAT(available_codes.code_count, '/', codes.code_count) as codes_available
        FROM {enrol_accesscodes_codesets} cs
        JOIN {user} u ON cs.creatorid = u.id
        LEFT JOIN (SELECT COUNT(ac.id) as code_count, ac.codesetid FROM {enrol_accesscodes_codes} ac
            GROUP BY ac.codesetid) codes ON codes.codesetid = cs.id
        LEFT JOIN (SELECT COUNT(ac.id) as code_count, ac.codesetid FROM {enrol_accesscodes_codes} ac
            WHERE ac.status = 'available' GROUP BY ac.codesetid) available_codes ON available_codes.codesetid = cs.id";

$countsql = "SELECT COUNT(*) FROM {enrol_accesscodes_codesets} cs";

$clauses = " WHERE 1 = 1 ";
$params = [];

if (empty($showall)) {
    $params['creatorid'] = $USER->id;
    $clauses .= " AND cs.creatorid = :creatorid ";
}

if (!empty($search)) {
    $params['search'] = '%' . $DB->sql_like_escape($search) . '%';
    $clauses .= " AND cs.name LIKE :search ";
}

$sql = $sql . $clauses . "GROUP BY cs.id, u.firstname, u.lastname, available_codes.code_count, codes.code_count ORDER BY cs.name";

$countsql = $countsql . $clauses;

$codesets = $DB->get_records_sql($sql, $params, $offset, $limit);

$totalcodesets = $DB->count_records_sql($countsql, $params);

$pagingbar = new \paging_bar($totalcodesets, $page, $limit, $thisurl->out());

$pagedata = [
    'codesets' => array_values($codesets),
    'plugin_url' => $CFG->wwwroot . '/enrol/accesscodes',
    'can_see_all' => ($hasaccessto == 'all'),
    'search_value' => $search ?? '',
    'show_all_value' => $showall,
];


$PAGE->set_title($heading);
$PAGE->set_heading($heading);
$PAGE->set_url($thisurl);
$PAGE->set_context($systemcontext);


echo $OUTPUT->header();

echo $OUTPUT->render_from_template('enrol_accesscodes/manage_codesets', $pagedata);

echo $OUTPUT->footer();
