<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Access codes external PHPunit tests
 *
 * @package    enrol_accesscodes
 * @copyright  2025 Andy McGill {@link https://brookesia.co.uk}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace enrol_accesscodes;

use externallib_advanced_testcase;
use enrol_accesscodes\external\get_instance_info;
use enrol_accesscodes\external\enrol_user;
use context_course;
use moodle_exception;

defined('MOODLE_INTERNAL') || die();

global $CFG;

require_once($CFG->dirroot . '/webservice/tests/helpers.php');

/**
 * Enrol accesscodes external functions tests
 *
 * @package    enrol_accesscodes
 * @category   external
 * @covers     \enrol_accesscodes\externallib
 * @copyright  2025 Andy McGill {@link https://brookesia.co.uk}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @since      Moodle 3.8
 * @group enrol_accesscodes_externalib
 */
final class external_test extends externallib_advanced_testcase {
    /**
     * test_get_instance_info description
     */
    public function test_get_instance_info(): void {
        global $DB;

        $this->resetAfterTest(true);

        // Check if accesscodes enrolment plugin is enabled.
        $accesscodesplugin = enrol_get_plugin('accesscodes');
        $this->assertNotEmpty($accesscodesplugin);

        $studentrole = $DB->get_record('role', ['shortname' => 'student']);
        $this->assertNotEmpty($studentrole);

        $coursedata = new \stdClass();
        $coursedata->visible = 0;
        $course = self::getDataGenerator()->create_course($coursedata);

        // Add enrolment methods for course.
        $instanceid1 = $accesscodesplugin->add_instance($course, [
            'status' => ENROL_INSTANCE_ENABLED,
            'name' => 'Test instance 1',
            'customint6' => 1,
            'roleid' => $studentrole->id,
        ]);
        $instanceid2 = $accesscodesplugin->add_instance($course, [
            'status' => ENROL_INSTANCE_DISABLED,
            'customint6' => 1,
            'name' => 'Test instance 2',
            'roleid' => $studentrole->id,
        ]);

        $enrolmentmethods = $DB->get_records('enrol', ['courseid' => $course->id, 'status' => ENROL_INSTANCE_ENABLED]);
        $this->assertCount(2, $enrolmentmethods);

        $this->setAdminUser();
        $instanceinfo1 = get_instance_info::execute($instanceid1);
        $instanceinfo1 = get_instance_info::clean_returnvalue(get_instance_info::execute_returns(), $instanceinfo1);

        $this->assertEquals($instanceid1, $instanceinfo1['id']);
        $this->assertEquals($course->id, $instanceinfo1['courseid']);
        $this->assertEquals('accesscodes', $instanceinfo1['type']);
        $this->assertEquals('Test instance 1', $instanceinfo1['name']);
        $this->assertTrue($instanceinfo1['status']);

        $instanceinfo2 = get_instance_info::execute($instanceid2);
        $instanceinfo2 = get_instance_info::clean_returnvalue(get_instance_info::execute_returns(), $instanceinfo2);
        $this->assertEquals($instanceid2, $instanceinfo2['id']);
        $this->assertEquals($course->id, $instanceinfo2['courseid']);
        $this->assertEquals('accesscodes', $instanceinfo2['type']);
        $this->assertEquals('Test instance 2', $instanceinfo2['name']);
        $this->assertEquals(get_string('cannotenrol', 'enrol_accesscodes'), $instanceinfo2['status']);

        // Try to retrieve information using a normal user for a hidden course.
        $user = self::getDataGenerator()->create_user();
        $this->setUser($user);
        try {
            get_instance_info::execute($instanceid2);
        } catch (moodle_exception $e) {
            $this->assertEquals('coursehidden', $e->errorcode);
        }
    }

    /**
     * test_enrol_user description
     */
    public function test_enrol_user(): void {
        global $DB;

        self::resetAfterTest(true);

        $user = self::getDataGenerator()->create_user();
        self::setUser($user);

        $course1 = self::getDataGenerator()->create_course();
        $course2 = self::getDataGenerator()->create_course(['groupmode' => SEPARATEGROUPS, 'groupmodeforce' => 1]);
        $user1 = self::getDataGenerator()->create_user();
        $user2 = self::getDataGenerator()->create_user();
        $user3 = self::getDataGenerator()->create_user();
        $user4 = self::getDataGenerator()->create_user();

        $context1 = context_course::instance($course1->id);
        $context2 = context_course::instance($course2->id);

        $accesscodesplugin = enrol_get_plugin('accesscodes');

        $enabled = enrol_get_plugins(true);
        $enabled['accesscodes'] = true;
        $enabled = array_keys($enabled);
        set_config('enrol_plugins_enabled', implode(',', $enabled));

        $studentrole = $DB->get_record('role', ['shortname' => 'student']);
        $instance1id = $accesscodesplugin->add_instance($course1, [
            'status' => ENROL_INSTANCE_ENABLED,
            'name' => 'Test instance 1',
            'customint6' => 1,
            'roleid' => $studentrole->id,
        ]);

        $instance2id = $accesscodesplugin->add_instance($course2, [
            'status' => ENROL_INSTANCE_DISABLED,
            'customint6' => 1,
            'name' => 'Test instance 2',
            'roleid' => $studentrole->id,
        ]);
        $instance1 = $DB->get_record('enrol', ['id' => $instance1id], '*', MUST_EXIST);
        $instance2 = $DB->get_record('enrol', ['id' => $instance2id], '*', MUST_EXIST);

        $codesetobject = (object) [
            'name' => 'Test codeset',
            'creatorid' => 2,
            'allcourses' => 1,
            'timecreated' => time(),
            'timemodified' => time(),

        ];
        $codesetid = $DB->insert_record('enrol_accesscodes_codesets', $codesetobject);

        $code1 = (object)[
            'code' => 'ABC123',
            'codesetid' => $codesetid,
            'status' => 'available',
        ];
        $code1->id = $DB->insert_record('enrol_accesscodes_codes', $code1);

        $code2 = (object)[
            'code' => 'ABC321',
            'codesetid' => $codesetid,
            'status' => 'available',
        ];
        $code2->id = $DB->insert_record('enrol_accesscodes_codes', $code2);

        self::setUser($user1);

        $result = enrol_user::execute($course1->id, $code1->code);
        $result = enrol_user::clean_returnvalue(enrol_user::execute_returns(), $result);

        self::assertTrue($result['status']);
        self::assertEquals(1, $DB->count_records('user_enrolments', ['enrolid' => $instance1->id]));
        self::assertTrue(is_enrolled($context1, $user1));

        // Try instance not enabled.
        try {
            enrol_user::execute($course2->id, $code2->code);
        } catch (moodle_exception $e) {
            self::assertEquals('cannotenrol', $e->errorcode);
        }

        // Enable the instance.
        $accesscodesplugin->update_status($instance2, ENROL_INSTANCE_ENABLED);

        // Try passing an invalid key.
        $result = enrol_user::execute($course2->id, 'invalidkey');
        $result = enrol_user::clean_returnvalue(enrol_user::execute_returns(), $result);
        self::assertFalse($result['status']);
        self::assertCount(1, $result['warnings']);
        self::assertEquals('4', $result['warnings'][0]['warningcode']);

        // Everything correct, now.
        $result = enrol_user::execute($course2->id, $code2->code);
        $result = enrol_user::clean_returnvalue(enrol_user::execute_returns(), $result);

        self::assertTrue($result['status']);
        self::assertEquals(1, $DB->count_records('user_enrolments', ['enrolid' => $instance2->id]));
        self::assertTrue(is_enrolled($context2, $user1));
    }
}
