<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Access codes enrolment plugin tests.
 *
 * @package    enrol_accesscodes
 * @copyright  2025 Andy McGill {@link https://brookesia.co.uk}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace enrol_accesscodes;

use advanced_testcase;
use null_progress_trace;
use progress_trace_buffer;
use text_progress_trace;
use context_course;

defined('MOODLE_INTERNAL') || die();

global $CFG;

require_once($CFG->dirroot . '/enrol/accesscodes/locallib.php');

/**
 * Test accesscodes plugin
 *
 * @covers     \enrol_accesscodes\locallib
 */
final class locallib_test extends advanced_testcase {
    /**
     * Test enrol_accesscodes_check_access_code
     */
    public function test_enrol_accesscodes_check_access_code(): void {
        global $DB;
        self::resetAfterTest(true);

        $user1 = $this->getDataGenerator()->create_user();

        // Test in course with groups.
        $course = self::getDataGenerator()->create_course(['groupmode' => SEPARATEGROUPS, 'groupmodeforce' => 1]);

        $accesscodegenerator = $this->getDataGenerator()->get_plugin_generator('enrol_accesscodes');

        $codeset = $accesscodegenerator->create_codeset(['name' => 'Test codeset', 'userid' => $user1->id]);

        $accesscode1 = $accesscodegenerator->create_code([
            'codesetid' => $codeset->id,
            'courseid' => $course->id,
            'code' => 'thecode',
        ]);

        $invalidcodeerrorstring = get_string("invalidcode", 'enrol_accesscodes');

        $result = enrol_accesscodes_check_access_code($course->id, 'invalidcode');
        $this->assertSame($invalidcodeerrorstring, $result);

        $result = enrol_accesscodes_check_access_code($course->id, 'thecode');
        $this->assertTrue($result);

        // Test disabling group options.
        $course->groupmode = NOGROUPS;
        $course->groupmodeforce = 0;
        $DB->update_record('course', $course);

        $result = enrol_accesscodes_check_access_code($course->id, 'invalidcode');
        $this->assertSame($invalidcodeerrorstring, $result);

        $result = enrol_accesscodes_check_access_code($course->id, 'thecode');
        $this->assertTrue($result);

        // Test course code is not valid for.
        $othercourse = self::getDataGenerator()->create_course();
        $result = enrol_accesscodes_check_access_code($othercourse->id, 'thecode');
        $invalidcourseerrorstring = get_string("invalidforcourse", 'enrol_accesscodes');
        $this->assertSame($invalidcourseerrorstring, $result);

        // Test codeset with expiry in the future.
        $futuretime = time() + 3600;
        $codeset2 = $accesscodegenerator->create_codeset(
            ['name' => 'Test codeset 2', 'userid' => $user1->id, 'timeexpires' => $futuretime]
        );
        $accesscode2 = $accesscodegenerator->create_code([
            'codesetid' => $codeset2->id,
            'courseid' => $course->id,
            'code' => 'thecode2',
        ]);

        $result = enrol_accesscodes_check_access_code($course->id, $accesscode2->code);
        $this->assertTrue($result);

        // Test codeset with expiry in the past.
        $pasttime = time() - 3600;
        $codeset3 = $accesscodegenerator->create_codeset(
            ['name' => 'Test codeset 3', 'userid' => $user1->id, 'timeexpires' => $pasttime]
        );
        $accesscode3 = $accesscodegenerator->create_code([
            'codesetid' => $codeset3->id,
            'courseid' => $course->id,
            'code' => 'thecode3',
        ]);

        $result = enrol_accesscodes_check_access_code($course->id, $accesscode3->code);
        $expiredcodeerrorstring = get_string("expiredcode", 'enrol_accesscodes');
        $this->assertSame($expiredcodeerrorstring, $result);
    }

    /**
     * Test enrol_accesscodes_enrol_form
     */
    public function test_enrol_accesscodes_form(): void {
        global $DB;
        self::resetAfterTest(true);

        $user = $this->getDataGenerator()->create_user();
        $course = self::getDataGenerator()->create_course();
        $studentrole = $DB->get_record('role', ['shortname' => 'student']);
        $accesscodesplugin = enrol_get_plugin('accesscodes');
        $accesscodesplugin->add_instance($course, ['status' => ENROL_INSTANCE_ENABLED, 'roleid' => $studentrole->id]);
        $instance = $DB->get_record('enrol', ['courseid' => $course->id, 'enrol' => 'accesscodes'], '*', MUST_EXIST);

        $accesscodegenerator = $this->getDataGenerator()->get_plugin_generator('enrol_accesscodes');

        $codeset = $accesscodegenerator->create_codeset(['name' => 'Test codeset', 'userid' => $user->id]);

        $accesscode1 = $accesscodegenerator->create_code([
            'codesetid' => $codeset->id,
            'courseid' => $course->id,
            'code' => 'thecode',
        ]);

        // Get the data we are submitting for the form.
        $formdata = [];
        $formdata['accesscode'] = $accesscode1->code;
        $formdata['id'] = $course->id;
        $formdata['instance'] = $instance->id;

        \enrol_accesscodes_enrol_form::mock_submit($formdata, [], 'post', $instance->id . '_enrol_accesscodes_enrol_form');

        $form = new \enrol_accesscodes_enrol_form(null, $instance);

        $data = $form->get_data();

        foreach ($formdata as $key => $value) {
            $this->assertEquals($value, $data->$key);
        }
        $this->assertTrue($form->is_validated());

        $formdata['accesscode'] = 'thewrongcode';
        \enrol_accesscodes_enrol_form::mock_submit($formdata, [], 'post', $instance->id . '_enrol_accesscodes_enrol_form');

        $form = new \enrol_accesscodes_enrol_form(null, $instance);

        $data = $form->get_data();

        $this->assertFalse($form->is_validated());
    }
}
