<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Class sync
 *
 * @package    enrol_campusonline
 * @copyright  2024, TU Graz
 * @author     think-modular (stefan.weber@think-modular.com)
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace enrol_campusonline;

use moodle_url;
use enrol_campusonline\locallib;
use GuzzleHttp\Client;

defined('MOODLE_INTERNAL') || die();

require_once($CFG->dirroot . '/user/lib.php');

/**
 * Class sync
 *
 * @package    enrol_campusonline
 * @copyright  2024, TU Graz
 * @author     think-modular (stefan.weber@think-modular.com)
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class sync {
    /**
     * String used for description who created courses/categories etc.
     * @var string
     */
    public const CREATED_BY = "Created by CAMPUSonline";

    /**
     * Mapping constants for course types.
     */
    public const GROUP_TO_COURSE = 'GROUP_TO_COURSE';

    /**
     * Mapping constant for group types.
     */
    public const GROUP_TO_GROUP = 'GROUP_TO_GROUP';

    /**
     * Mapping constant for flat course types.
     */
    public const FLAT_COURSE = 'FLAT_COURSE';

    /**
     * The Moodle config of the CAMPUSonline plugin.
     * @var object
     */

    private $config;
    /**
     * Stores the error message if there has been an error.
     * @var string
     */
    private $error;

    /**
     * The current CAMPUSonline access token
     * @var string
     */
    private $token;

    /**
     * The Moodle progress trace, outputs to plain text.
     * @var \text_progress_trace
     */
    private $trace;

    /**
     * External key (TODO: what is it?)
     * @var string
     */
    private $externalkey;

    /**
     * External system key (TODO: what is it?)
     * @var string
     */
    private $externalsystemkey;

    /**
     * Summary of customfieldids (TODO: custom field id's within Moodle or CAMPUSonline?)
     * @var array
     */
    private $customfieldids;

    /**
     * Organizations data
     * @var array
     */
    private $orgdata;

    /**
     * Organization roles data (keys: roleid, values: name).
     * @var array
     */
    private $orgroles;

    /**
     * Creation time of the OpenID token.
     * @var int
     */
    private $tokencreation;

    /**
     * Maximum age of the OpenID token with a safety buffer.
     * @var int
     */
    private $maxtokenage;

    /**
     * Timeout for API responses in seconds.
     * @var float
     */
    private $connect;

    /**
     * Timeout for API connections in seconds.
     * @var float
     */
    private $connecttimeout;

    /**
     * Semester data from CAMPUSonline (key: CAMPUSonline-Semester-Key, value: Semester-Object).
     * @var array
     */
    private $semesterdata;

    /**
     * Array of group-to-group mappings from the CAMPUSonline plugin configuration.
     *
     * Will be false, if plugin configuration is malicious.
     *
     * @var string[]|false
     */
    private $grouptogroup;

    /**
     * Array of group-to-course mappings from the CAMPUSonline plugin configuration.
     *
     * Will be false, if plugin configuration is malicious.
     *
     * @var string[]|false
     */
    private $grouptocourse;

    /**
     * Array of flatcourse mappings from the CAMPUSonline plugin configuration.
     *
     * List of e-learning Event types for which Moodle courses will be created, but groups will be ignored.
     * Will be false, if plugin configuration is malicious.
     *
     * @var string[]|false
     */
    private $flatcourse;

    /**
     * Constructor.
     *
     * @param \text_progress_trace $trace
     */
    public function __construct($trace) {

        global $DB;

        // Do nothing if our enrolment method is disabled.
        if (enrol_is_enabled('campusonline')) {
            // Remove old logs.
            locallib::cleanup_logs();

            // Get settings.
            $this->config = get_config('enrol_campusonline');
            $this->trace = $trace;
            $this->externalkey = $this->config->user_externalkey;
            $this->externalsystemkey = $this->config->user_externalsystemkey;
            $this->grouptocourse = preg_split('/\s*,\s*/', $this->config->grouptocourse);
            $this->grouptogroup = preg_split('/\s*,\s*/', $this->config->grouptogroup);
            $this->flatcourse = preg_split('/\s*,\s*/', $this->config->flatcourse);
            // Client class expects floats for these values.
            $this->timeout = (float) $this->config->timeout;
            $this->connecttimeout = (float) $this->config->connect_timeout;
            // Remove a small 5 second buffer from the config max token age value.
            if ($this->config->maxtokenage > 5) {
                $this->maxtokenage = $this->config->maxtokenage - 5;
            } else {
                $this->maxtokenage = $this->config->maxtokenage;
            }

            // Get roles to sync for organisations.
            $this->orgroles = locallib::get_org_roles();

            // Get custom field ids so we dont have to deal with Moodle custom field API.
            $fields = ['user_info_field:campusonline_person_uid',
                    'customfield_field:campusonline_other_co_course_uids',
                    ];
            foreach ($fields as $field) {
                [$table, $shortname] = explode(':', $field);
                if (!$value = $DB->get_field($table, 'id', ['shortname' => $shortname])) {
                    // Show error.
                    $message = get_string('error:uidfieldnotfound', 'enrol_campusonline', "$table: $shortname");
                    \core\notification::add(
                        $message,
                        \core\output\notification::NOTIFY_ERROR
                    );
                    locallib::write_log('general', $message, 2, null, $this->trace, null);
                } else {
                    $this->customfieldids[$shortname] = $value;
                }
            }

            // Get token.
            $this->update_token();
        }
    }

    /**
     * Checks if connection was successful.
     */
    public function is_connected() {
        return !empty($this->token);
    }

    /**
     * Returns the error message.
     */
    public function get_error() {
        return $this->error;
    }

    /**
     * Gets a category for a course.
     *
     * @param array $coursedata
     *
     * @return string $categoryid
     */
    public function get_course_category($coursedata) {

        global $DB;

        // Get category for this course's org.
        $orgid = $coursedata['org:uid'];
        if ($orgcategory = $DB->get_record('course_categories', ['idnumber' => $orgid])) {
            $categoryid = $orgcategory->id;
        } else {
            // Log warning.
            $message = get_string('warning:orgcategorynotfound', 'enrol_campusonline', $orgid);
            $categoryid = $this->config->rootcoursecategory;
        }

        // Get subcategories.
        $subcategories = $this->config->subcategories;
        $subcategories = explode('\\', $subcategories);

        foreach ($subcategories as $name) {
            foreach ($coursedata as $key => $value) {
                if (is_string($value)) {
                    $name = str_replace('{' . $key . '}', $value, $name);
                }
            }
            $category = $DB->get_record('course_categories', ['name' => $name, 'parent' => $categoryid]);

            if (!$category) {
                if ($this->config->createcoursecategories == 0) {
                    // Log error.
                    $message = get_string('error:coursecategorynotfoundandcreationdisabled', 'enrol_campusonline', $name);
                    locallib::write_log('create_category', $message, 2, null, $this->trace, 3);

                    return false;
                } else {
                    // Log creation.
                    $message = get_string('info:coursecategorycreated', 'enrol_campusonline', $name);
                    locallib::write_log('create_category', $message, 0, null, $this->trace, 3);

                    // Create new category.
                    $categorydata = new \stdClass();
                    $categorydata->name = $name;
                    $categorydata->parent = $categoryid;
                    $categorydata->description = self::CREATED_BY;
                    $category = \core_course_category::create($categorydata);
                    $categoryid = $category->id;
                }
            } else {
                $category = $DB->get_record('course_categories', ['name' => $name, 'parent' => $categoryid]);
                $categoryid = $category->id;
            }
        }

        return $categoryid;
    }

    /**
     * Gets course description.
     *
     * @param string $courseuid
     *
     * @return array $descriptions
     */
    public function get_course_descriptions($courseuid) {

        $endpoint = "co-tm-core/course/api/course-descriptions";
        $query = [
            'course_uid' => $courseuid,
        ];
        $result = $this->rest_call($endpoint, $query);

        // Analyze response.
        $descriptions = [];
        if (property_exists($result, 'items')) {
            foreach ($result->items as $item) {
                unset($item->uid);
                $item = (array)$item;
                $descriptions = $item;
            }
        }

        return $descriptions;
    }

    /**
     * Gets groups for a course.
     *
     * @param string $courseuid
     * @return array groups
     */
    public function get_course_groups($courseuid) {

            $endpoint = "co-tm-core/course/api/courses/$courseuid/groups";
            $result = $this->rest_call($endpoint, null);

        // Analyze response.
        $groups = [];
        if (property_exists($result, 'items')) {
            foreach ($result->items as $item) {
                $groups[$item->uid] = $item->name->value->de;
            }
        }

        return $groups;
    }

    /**
     * Gets courses for preview or sync.
     *
     * @param string $courseuids If provided, only fetches these courses.
     * @param int $limit
     * @return array
     */
    public function get_courses($courseuids = null, $limit = null) {

        $allcourses = [];

        // Get semester(s).
        $semesters = $this->config->semester;
        $semesters = explode(',', $semesters);

        foreach ($semesters as $semester) {
            $semester = trim($semester);

            // Get courses.
            $endpoint = 'co-tm-core/course/api/courses';
            if ($courseuids) {
                // Only request specific courses.
                $query['course_uids'] = implode(', ', $courseuids);
            } else {
                // Request all courses for configured semester(s).
                $query = [
                    'semester_key' => $semester,
                    'only_elearning_courses' => 'true',
                    'limit' => $limit,
                ];
            }

            $result = $this->rest_call($endpoint, $query);

            // Analyze response.
            if (property_exists($result, 'items')) {
                $courses = $result->items;
                $courses = $this->enrich_courses($courses);
            } else {
                $courses = [];
            }

            $allcourses = array_merge($allcourses, $courses);

            // Stop if we specified course_uids, since semester filter will be ignored anyways.
            if ($courseuids) {
                break;
            }
        }

        return $allcourses;
    }

    /**
     * Gets course sync strategy for a course.
     *
     * @param array $coursedata The course data array containing course information.
     * @return string|null Returns the course type constant or null if not found.
     */
    public function get_course_sync_strategy($coursedata) {

        $courseuid = $coursedata['course:uid'];

        // Get eLearningEventTypeKey.
        if (array_key_exists('course:elearningEventTypeKey', $coursedata)) {
            $elearningtype = $coursedata['course:elearningEventTypeKey'];
        } else {
            // Log error.
            $message = get_string('error:noelearningeventtypekey', 'enrol_campusonline', $courseuid);
            locallib::write_log('update_course', $message, 2, null, $this->trace, 3);
            return null;
        }

        if (in_array($elearningtype, $this->grouptocourse)) {
            return self::GROUP_TO_COURSE;
        } else if (in_array($elearningtype, $this->grouptogroup)) {
            return self::GROUP_TO_GROUP;
        } else if (in_array($elearningtype, $this->flatcourse)) {
            return self::FLAT_COURSE;
        } else {
            if (PHP_SAPI == 'cli' || array_key_exists('traceoutput', $_GET)) {
                $this->trace->output(
                    get_string(
                        'info:skippingcourse',
                        'enrol_campusonline',
                        [
                            'course_uid' => $courseuid,
                            'elearning_type' => $elearningtype,
                        ]
                    )
                );
            }
            return null;
        }
    }

    /**
     * Gets enrolments for a course from CAMPUSonline.
     *
     * @param object $course
     *
     * @return array $enrolments
     */
    public function get_enrolments($course) {

        $uids = explode(':', $course->idnumber);
        $courseuid = $uids[0];
        if (count($uids) > 1) {
            $groupuid = $uids[1];
        }

        // Get student enrolments.
        $endpoint = 'co-tm-core/course/api/registrations';
        $query = [
            'course_uid' => $courseuid,
        ];
        $result = $this->rest_call($endpoint, $query);

        // Analyze response.
        if (property_exists($result, 'items')) {
            $enrolments = $result->items;
        } else {
            $enrolments = [];
        }

        // Get teacher enrolments.
        $endpoint = 'co-tm-core/course/api/lectureships';
        $query = [
            'course_uid' => $courseuid,
        ];
        $result = $this->rest_call($endpoint, $query);

        // Analyze response.
        if (property_exists($result, 'items')) {
            $enrolments = array_merge($result->items, $enrolments);
        }

        return $enrolments;
    }

    /**
     * Gets lectureship functions from CAMPUSonline.
     *
     * @return array $lectureship_functions
     */
    public function get_lectureship_functions() {
        $endpoint = 'co-tm-core/course/api/lectureship-functions';
        $result = $this->rest_call($endpoint);

        // Analyze response.
        $lectureshipfunctions = [];
        if (property_exists($result, 'items')) {
            foreach ($result->items as $item) {
                $key = $item->key;
                $name = locallib::normalize_value($item->name);
                $lectureshipfunctions[$key] = "$name ($key)";
            }
        }

        return $lectureshipfunctions;
    }

    /**
     * Gets the Moodle User ID of a CAMPUSonline user via its uid.
     *
     * @param string $uid
     * @param bool $log
     * @param string|null $usertype Optional type of user, used for config option separateusers.
     *
     * @return mixed $userid
     */
    public function get_moodle_user_id($uid, $log = true, $usertype = null) {

        global $DB;
        $userid = null;

        // If configured, create separate users for staff and students.
        if ($this->config->separateusers && $usertype) {
            $uid = $usertype . '_' . $uid;
        }

        // Get user id via uid in our user profile field.
        $fieldid = $this->customfieldids['campusonline_person_uid'];
        $sql = "SELECT * FROM {user_info_data} WHERE fieldid = ? AND data = ?";
        $params = ['fieldid' => $fieldid, 'data' => $uid];
        if ($records = $DB->get_records_sql($sql, $params)) {
            $record = reset($records);
            return $record->userid;
        }

        // Log warning.
        if ($log) {
            $message = get_string('warning:moodleusernotfound', 'enrol_campusonline', $uid);
            locallib::write_log('get_user', $message, 1, null, $this->trace);
        }

        return null;
    }

    /**
     * Gets organitional roles from CAMPUSonline.
     *
     * @return array $orgroles
     */
    public function get_org_roles() {
        $endpoint = '/auth/api/role-names';
        $query = [
            'role_namespace' => 'F',
        ];

        $result = $this->rest_call($endpoint, $query);

        // Analyze response.
        $orgroles = [];
        if (property_exists($result, 'items')) {
            foreach ($result->items as $item) {
                $key = $item->name;
                $name = locallib::normalize_value($item->label);
                $orgroles[$key] = $name;
            }
        }

        return $orgroles;
    }

    /**
     * Gets additional person data from CAMPUSonline.
     *
     * @param string $uid
     * @param bool $log
     *
     * @return array $persondata
     */
    public function get_person_data($uid, $log = true) {

        $endpoint = "co-brm-core/pers/api/person-claims";
        $query = [
            'claim' => 'CO_CLAIM_ALL',
            'person_uid' => $uid,
        ];

        $result = $this->rest_call($endpoint, $query);

        // Log error.
        if (!property_exists($result, 'items') || empty($result->items)) {
            if ($log) {
                $message = get_string('warning:couldnotgetpersondata', 'enrol_campusonline', $uid);
                locallib::write_log('get_user_data', $message, 1, null, $this->trace, 3);
            }
            return [];
        } else {
            $persondata = (array)$result->items[0];
        }

        return $persondata;
    }

    /**
     * Gets external keys for person for user identification.
     *
     * @param string $uid
     *
     * @return array
     */
    public function get_person_external_key($uid) {

        $endpoint = "co-brm-core/pers/api/person-identifiers/mappings";
        $query = [
            'source_claim' => 'CO_CLAIM_PERSON_UID',
            'target_claim' => 'CO_CLAIM_EXTERNAL_SYSTEM_UID',
            'uid' => $uid,
            'external_key' => $this->externalkey,
            'external_system_key' => $this->externalsystemkey,
        ];

        $result = $this->rest_call($endpoint, $query);

        if (property_exists($result, 'mappings')) {
            return (array)$result->mappings;
        } else {
            return [];
        }
    }

    /**
     * Gets persons from CAMPUSonline.
     *
     * @param string[] $personuids
     * @param int $limit
     *
     * @return array $persons
     */
    public function get_persons($personuids = null, $limit = null) {

        // Get employees.
        $endpoint = "co-brm-core/pers/api/person-claims";
        $query = [
            'claim' => 'CO_CLAIM_ALL',
            'limit' => $limit,
        ];

        if ($personuids) {
            $query['person_uid'] = implode(',', $personuids);
        }

        $result = $this->rest_call($endpoint, $query);

        // Analyze response.
        if (property_exists($result, 'items')) {
            $personobjects = $result->items;
        } else {
            $personobjects = [];
        }

        $persons = [];
        foreach ($personobjects as $personobject) {
            $persons[] = (array) $personobject;
        }

        return $persons;
    }

    /**
     * Maps CAMPUSonline persons to existing Moodle users and updates their person UID.
     *
     * @param array $users
     *
     */
    public function identify_moodle_users($users) {

        global $CFG, $DB;

        require_once($CFG->dirroot . '/user/profile/lib.php');

        $sourcefield = $this->config->sourcefield;
        $sourceclaim = $this->config->sourceclaim;
        $maxattempts = (int) $this->config->idattempts;
        if (str_contains($sourcefield, 'profile_field_')) {
            $sourcefield = str_replace('profile_field_', '', $sourcefield);
            $profilefield = true;
        } else {
            $profilefield = false;
        }

        $total = count($users);
        $skipped = 0;
        foreach ($users as $key => $user) {
            // Skip users that already have a Person UID.
            if (locallib::get_person_uid($user->id)) {
                unset($users[$key]);
                $skipped++;
            }
        }

        $number = count($users);
        if (PHP_SAPI == 'cli' || array_key_exists('traceoutput', $_GET)) {
            if ($number > 1) {
                $this->trace->output(
                    get_string(
                        'info:identifyingusers',
                        'enrol_campusonline',
                        [
                            'number' => $number,
                            'total' => $total,
                            'skipped' => $skipped,
                        ]
                    )
                );
            } else {
                $this->trace->output(get_string('info:identifyinguser', 'enrol_campusonline'));
            }
        }

        foreach ($users as $user) {
            $userid = $user->id;
            if (PHP_SAPI == 'cli' || array_key_exists('traceoutput', $_GET)) {
                $this->trace->output(get_string('info:identifyinguserwithid', 'enrol_campusonline', $userid));
            }
            profile_load_custom_fields($user);

            // Skip users that have reached the maximum attempts.
            if (array_key_exists('campusonline_id_attempts', $user->profile)) {
                $attempt = (int) $user->profile['campusonline_id_attempts'];
                if ($attempt >= $maxattempts) {
                    if (PHP_SAPI == 'cli' || array_key_exists('traceoutput', $_GET)) {
                        $this->trace->output(
                            get_string(
                                'info:maxattemptsreached',
                                'enrol_campusonline',
                                $userid
                            )
                        );
                    }
                    continue;
                }
            }

            // Get uid value.
            $uid = null;
            if ($profilefield) {
                if (property_exists($user, 'profile')) {
                    if (array_key_exists($sourcefield, $user->profile)) {
                        $uid = $user->profile[$sourcefield];
                    }
                }
            } else {
                $uid = $user->$sourcefield;
            }

            // Skip users that do not have an uid value.
            if (!$uid) {
                // Update attempts.
                $user->profile_field_campusonline_id_attempts = $attempt + 1;
                $result = profile_save_data($user);

                // Log.
                $message = get_string(
                    'info:couldnotfinduidvalue',
                    'enrol_campusonline',
                    [
                        'sourcefield' => $sourcefield,
                        'userid' => $userid,
                        'attempt' => $attempt,
                    ]
                );
                locallib::write_log('identify_user', $message, 0, null, $this->trace, 3);
                continue;
            }

            // Skip users that have a uid that is already set to another Moodle user.
            if ($moodleuserid = $this->get_moodle_user_id($uid, false)) {
                $message = get_string(
                    'warning:uidalreadyassigned',
                    'enrol_campusonline',
                    [
                        'uid' => $uid,
                        'moodle_user_id' => $moodleuserid,
                        'userid' => $userid,
                    ]
                );
                locallib::write_log('identify_user', $message, 1, null, $this->trace, 3);
                continue;
            }

            // If source claim equals target claim, that means we already have the Person UID,
            // and can try fetching the person to see if it is valid.
            if ($this->config->sourceclaim == 'CO_CLAIM_PERSON_UID') {
                if ($this->get_person_data($uid, false)) {
                    // Save Person UID to user profile.
                    $user->profile_field_campusonline_person_uid = $uid;
                    profile_save_data($user);

                    // Log success.
                    $message = get_string('info:useridentified', 'enrol_campusonline', ['userid' => $userid, 'uid' => $uid]);
                    locallib::write_log('identify_user', $message, 0, null, $this->trace, 3);
                    continue;
                }
            } else {
                // Fetch Person UID using the source claim.
                $endpoint = 'co-brm-core/pers/api/person-identifiers/mappings';
                $query = [
                    'target_claim' => 'CO_CLAIM_PERSON_UID',
                    'source_claim' => $sourceclaim,
                    'uid' => $uid,
                ];
                if ($this->config->sourceclaim == 'CO_CLAIM_EXTERNAL_SYSTEM') {
                    $query['external_key'] = $this->config->externalkey;
                    $query['external_system_key'] = $this->config->externalsystemkey;
                }
                $result = $this->rest_call($endpoint, $query);

                // Analyze response.
                if (property_exists($result, 'mappings')) {
                    $mapping = $result->mappings;
                    if (property_exists($mapping, $uid)) {
                        $personuid = $mapping->$uid;

                        // Save Person UID to user profile.
                        $user->profile_field_campusonline_person_uid = $personuid;
                        profile_save_data($user);

                        // Log success.
                        $message = get_string(
                            'info:useridentifiedvia',
                            'enrol_campusonline',
                            [
                                'userid' => $userid,
                                'person_uid' => $personuid,
                                'sourceclaim' => $sourceclaim,
                                'uid' => $uid,
                            ]
                        );
                        locallib::write_log('identify_user', $message, 0, null, $this->trace, 3);
                        continue;
                    }
                }
            }

            // Update attempts.
            $user->profile_field_campusonline_id_attempts = $attempt + 1;
            profile_save_data($user);

            // Log.
            $message = get_string(
                'info:couldnotfindperson',
                'enrol_campusonline',
                [
                    'uid' => $uid,
                    'userid' => $userid,
                    'attempt' => $attempt,
                ]
            );
            locallib::write_log('identify_user', $message, 0, null, $this->trace, 3);
            continue;
        }
    }

    /**
     * Syncs modified courses.
     *
     * @return void
     */
    public function sync_course_delta() {
        // Set timeframe.
        $timeframe = $this->config->modificationtimeframe;

        // Get todays date minus timeframe days.
        $date = new \DateTime();
        $date->sub(new \DateInterval('P' . $timeframe . 'D'));
        $date = $date->format('Y-m-d');

        // Get semester(s).
        $semesters = $this->config->semester;
        $semesters = explode(',', $semesters);

        // Get modified courses.
        $courseuids = [];
        $components = ['courses', 'registrations', 'lectureships'];
        foreach ($semesters as $semester) {
            foreach ($components as $component) {
                $endpoint = "co-tm-core/course/api/$component/modifications";
                $query = [
                    'since' => $date,
                    'semestery_key' => $semester,
                ];
                $result = $this->rest_call($endpoint, $query);
                if (property_exists($result, 'items')) {
                    foreach ($result->items as $item) {
                        $courseuids[$item->courseUid] = 'modified';
                    }
                }
            }
        }

        // Sync modified courses.
        $courseuids = array_keys($courseuids);
        foreach ($courseuids as $courseuid) {
            $this->sync_courses([$courseuid]);
        }
    }

    /**
     * Syncs courses.
     *
     * @param string $courseuids if provided, only syncs these courses.
     * @return void
     */
    public function sync_courses($courseuids = null) {
        global $CFG, $DB;

        require_once("$CFG->dirroot/course/lib.php");

        // Get course(s).
        if ($courseuids) {
            $courses = $this->get_courses($courseuids);
        } else {
            $courses = $this->get_courses();
        }

        // Start output.
        $number = count($courses);
        if (PHP_SAPI == 'cli' || array_key_exists('traceoutput', $_GET)) {
            $this->trace->output(get_string('info:syncingcourses', 'enrol_campusonline', $number));
        }

        // Sync courses.
        foreach ($courses as $coursedata) {
            $courseuid = $coursedata['course:uid'];

            // Skip courses that are not in the configured orgs.
            if ($orgfilter = $this->config->orgfilter) {
                $orgs = explode(',', $orgfilter);
                if (!in_array($coursedata['org:uid'], $orgs)) {
                    if (PHP_SAPI == 'cli' || array_key_exists('traceoutput', $_GET)) {
                        $this->trace->output(get_string('info:skippingcourseorgfilter', 'enrol_campusonline', $courseuid));
                    }
                    continue;
                }
            }

            // Determine course sync strategy.
            if (!$strategy = self::get_course_sync_strategy($coursedata)) {
                continue;
            }

            // Start sync & log.
            if (PHP_SAPI == 'cli' || array_key_exists('traceoutput', $_GET)) {
                $this->trace->output('----------------------------------------------------------------');
                $this->trace->output(
                    get_string(
                        'info:syncingcourse',
                        'enrol_campusonline',
                        [
                            'course_uid' => $courseuid,
                            'strategy' => $strategy,
                        ]
                    )
                );
                $this->trace->output('----------------------------------------------------------------');
            }

            // Get groups if necessary.
            if ($strategy !== self::FLAT_COURSE) {
                $groups = $this->get_course_groups($courseuid);
            } else {
                $groups = [0 => 'dummy'];
            }

            foreach ($groups as $groupuid => $groupname) {
                // Prepare new course data.
                if ($strategy == self::GROUP_TO_COURSE) {
                    $idnumber = "$courseuid:$groupuid";
                    $newcourse = locallib::build_course($coursedata, $groupname, $groupuid);
                } else {
                    $idnumber = $courseuid;
                    $newcourse = locallib::build_course($coursedata);
                }

                // Get category (only for first loop).
                if (!$newcourse['category'] = $this->get_course_category($coursedata)) {
                    continue;
                }

                // Try to find existing course.
                if (!$course = $DB->get_record('course', ['idnumber' => $idnumber])) {
                    // Create new course.
                    $course = $this->create_course($newcourse, $coursedata, $strategy, $courseuid, $groupuid);
                } else {
                    // Update existing course.
                    $this->update_course($course, $newcourse, $coursedata, $strategy, $courseuid, $groupuid, $courseuids);
                }

                // Sync enrolments.
                if ($strategy == self::GROUP_TO_GROUP) {
                    $this->sync_enrolments($course, $groups);
                } else {
                    $this->sync_enrolments($course);
                }

                // Break foreach loop in case of no separate groups.
                if ($strategy !== self::GROUP_TO_COURSE) {
                    break;
                }
            }
        }
    }

    /**
     * Creates a Moodle course.
     *
     * @param array $newcourse
     * @param array $coursedata
     * @param string $strategy
     * @param string $courseuid
     * @param string|null $groupuid
     * @return object $course
     */
    private function create_course($newcourse, $coursedata, $strategy, $courseuid, $groupuid = null) {
        global $CFG, $DB;
        if ($templateid = $this->get_template_course_id($coursedata)) {
            // Create course from template.
            require_once("$CFG->dirroot/course/externallib.php");
            $coursearray = \core_course_external::duplicate_course(
                $templateid,
                $newcourse['fullname'],
                $newcourse['shortname'],
                $newcourse['category']
            );
            $courseid = $coursearray['id'];
            $course = $DB->get_record('course', ['id' => $courseid]);
            foreach ($newcourse as $key => $value) {
                $course->$key = $value;
            }
            $DB->update_record('course', $course);
        } else {
            // Create empty course.
            $course = new \stdClass();
            foreach ($newcourse as $key => $value) {
                $course->$key = $value;
            }
            $course = create_course($course);
        }

        // Log success.
        if ($course) {
            $courseid = $course->id;

            // Log success.
            if ($strategy == self::GROUP_TO_COURSE) {
                $this->set_moodle_course_url($course, $groupuid);
                $message = get_string(
                    'info:createdmoodlecoursegroup',
                    'enrol_campusonline',
                    [
                        'courseid' => $courseid,
                        'course_uid' => $courseuid,
                        'group_uid' => $groupuid,
                    ]
                );
            } else {
                $this->set_moodle_course_url($course);
                $message = get_string(
                    'info:createdmoodlecourse',
                    'enrol_campusonline',
                    [
                        'courseid' => $courseid,
                        'course_uid' => $courseuid,
                    ]
                );
            }
            // Add template id to log message.
            if ($templateid) {
                $message .= ' ' . get_string(
                    'info:createdfromtemplate',
                    'enrol_campusonline',
                    $templateid
                );
            }
            locallib::write_log('create_course', $message, 0, $courseid, $this->trace, 3);
        } else {
            // Log error.
            if ($strategy == self::GROUP_TO_COURSE) {
                $message = get_string(
                    'error:couldnotcreatemoodlecoursegroup',
                    'enrol_campusonline',
                    [
                        'course_uid' => $courseuid,
                        'group_uid' => $groupuid,
                    ]
                );
            } else {
                $message = get_string('error:couldnotcreatemoodlecourse', 'enrol_campusonline', $courseuid);
            }
            locallib::write_log('create_course', $message, 2, null, $this->trace, 3);
        }

        // Add custom fields.
        locallib::set_custom_course_fields($courseid, $coursedata);

        // Add our enrolment method.
        locallib::add_enrolment_method($course);

        return $course;
    }

    /**
     * Gets template course.
     *
     * @param array $coursedata
     * @return int|null $templatecourseid
     */
    private function get_template_course_id($coursedata) {
        global $DB;
        for ($i = 1; $i < 4; $i++) {
            $setting = 'coursetemplate' . $i;
            if (!$shortnameraw = $this->config->$setting) {
                continue;
            }
            $shortname = locallib::replace_tokens($shortnameraw, $coursedata);
            if ($shortname) {
                if ($template = $DB->get_record('course', ['shortname' => $shortname])) {
                    return $template->id;
                }
            }
        }
        return null;
    }

    /**
     * Updates a Moodle course.
     *
     * @param object $course
     * @param array $newcourse
     * @param array $coursedata
     * @param string $strategy
     * @param string $courseuid
     * @param string|null $groupuid
     * @param string[]|null $courseuids
     * @return void
     */
    private function update_course($course, $newcourse, $coursedata, $strategy, $courseuid, $groupuid = null, $courseuids = null) {
        global $DB;

        $courseid = $course->id;

        // Update course URL in CAMPUSonline.
        if ($this->config->updatecourseurls == 1) {
            if ($strategy == self::GROUP_TO_COURSE) {
                $this->set_moodle_course_url($course, $groupuid);
            } else {
                $this->set_moodle_course_url($course);
            }
        }

        // Skip.
        if (!$courseuids && $this->config->updateexistingcourses == 0) {
            return;
        }

        // Update course.
        $needsupdate = false;
        foreach ($newcourse as $key => $value) {
            if (property_exists($course, $key) && $course->$key != $value) {
                $course->$key = $value;
                $needsupdate = true;
            }
        }

        // Update course custom fields.
        if (locallib::set_custom_course_fields($courseid, $coursedata)) {
            $needsupdate = true;
        }

        if ($needsupdate) {
            $DB->update_record('course', $course);

            // Log success.
            if ($strategy == self::GROUP_TO_COURSE) {
                $message = get_string(
                    'info:updatedmoodlecoursegroup',
                    'enrol_campusonline',
                    ['courseid' => $courseid, 'course_uid' => $courseuid, 'group_uid' => $groupuid]
                );
            } else {
                $message = get_string(
                    'info:updatedmoodlecourse',
                    'enrol_campusonline',
                    ['courseid' => $courseid, 'course_uid' => $courseuid]
                );
            }
            locallib::write_log('update_course', $message, 0, $courseid, $this->trace, 3);
        }
    }

    /**
     * Syncs enrolments.
     *
     * @param object $course
     * @param array $groups
     *
     * @return void
     */
    public function sync_enrolments($course, $groups = null) {

        global $CFG, $DB;
        require_once("$CFG->dirroot/user/lib.php");
        require_once("$CFG->dirroot/group/lib.php");

        $courseid = $course->id;
        $context = \context_course::instance($courseid);

        // Get or create grouping.
        if (
            $grouping = $DB->get_record(
                'groupings',
                ['courseid' => $courseid,
                'name' => 'CAMPUSonline',
                'idnumber' => $course->idnumber,
                ]
            )
        ) {
            $groupingid = $grouping->id;
        } else {
            $grouping = new \stdClass();
            $grouping->courseid = $courseid;
            $grouping->name = 'CAMPUSonline';
            $grouping->idnumber = $course->idnumber;
            $grouping->timecreated = time();
            $grouping->timemodified = time();
            $groupingid = groups_create_grouping($grouping);
            $course->defaultgroupingid = $groupingid;
            $DB->update_record('course', $course);
            $message = get_string('info:createdgrouping', 'enrol_campusonline', $courseid);
            locallib::write_log('sync_groups', $message, 0, $courseid, $this->trace, 3);
        }

        // Check if enrolment method is active.
        if (!$enrol = $DB->get_record('enrol', ['courseid' => $courseid, 'enrol' => 'campusonline', 'status' => 0])) {
            $message = get_string('warning:skippingenrolments', 'enrol_campusonline', $courseid);
            locallib::write_log('enrol_user', $message, 1, $courseid, $this->trace, 3);
            return;
        }

        // Sum up existing enrolments in Moodle course, to save on DB queries.
        $existingenrolments = [];
        $existingenrolmentsraw = $DB->get_records('user_enrolments', ['enrolid' => $enrol->id]);
        foreach ($existingenrolmentsraw as $existingenrolment) {
            $existingenrolments[$existingenrolment->userid] = $existingenrolment;
        }

        // Get enrolments from CAMPUSonline.
        $assignedroles = [];
        $coenrolments = $this->get_enrolments($course);

        // Check if this is a course for a single groups.
        $uids = explode(':', $course->idnumber);
        $courseuid = $uids[0];
        $groupuid = null;
        if (count($uids) > 1) {
            $groupuid = $uids[1];
        }

        $groupmembers = [];
        foreach ($coenrolments as $coenrolment) {
            // Skip if enrolment is not for this group.
            if ($groupuid) {
                if (property_exists($coenrolment, 'courseGroupUid') && $coenrolment->courseGroupUid != $groupuid) {
                    continue;
                }
            }

            // Get role.
            if (property_exists($coenrolment, 'functionKey')) {
                $usertype = 'staff';
                $rolekey = 'role_' . $coenrolment->functionKey;
                if (property_exists($this->config, $rolekey)) {
                    $roleid = $this->config->$rolekey;
                } else {
                    continue;
                }
            } else {
                $usertype = 'student';
                $roleid = $this->config->studentrole;
            }

            // Skip if role should not be synced.
            if (!$roleid || $roleid == 0) {
                continue;
            }

            // Only log errors finding users if enrolment sync is not allowed to create users.
            $log = !$this->config->enrolsynccreateusers;

            // Get Moodle user.
            $uid = $coenrolment->personUid;
            $userid = $this->get_moodle_user_id($uid, $log, $usertype);

            // Create new user if needed & allowed.
            if (!$userid) {
                if ($this->config->enrolsynccreateusers) {
                    if (!$userid = $this->create_moodle_user($uid, $usertype)) {
                        continue;
                    }
                } else {
                    // Log warning.
                    $message = get_string('warning:skippingenrolment', 'enrol_campusonline', $uid);
                    locallib::write_log('enrol_user', $message, 1, $courseid, $this->trace, 3);
                    continue;
                }
            }

            // Create enrolment if needed.
            if (!in_array($userid, array_keys($existingenrolments))) {
                // Create enrolment.
                $enrolment = new \stdClass();
                $enrolment->enrolid = $enrol->id;
                $enrolment->userid = $userid;
                $enrolment->timestart = time();
                $enrolment->timeend = 0;
                $enrolment->status = 0;
                $enrolment->modifierid = 0;
                $enrolment->timecreated = time();
                $enrolment->timemodified = time();
                $DB->insert_record('user_enrolments', $enrolment);
                $existingenrolments[$userid] = $enrolment;

                // Log success.
                $message = get_string('info:enrolleduser', 'enrol_campusonline', ['userid' => $userid, 'courseid' => $courseid]);
                locallib::write_log('enrol_user', $message, 0, $courseid, $this->trace, 3);
            } else {
                // Activate enrolment if needed.
                if ($existingenrolments[$userid]->status == 1) {
                    $enrolment = $existingenrolments[$userid];
                    $enrolment->status = 0;
                    $enrolment->timemodified = time();
                    $DB->update_record('user_enrolments', $enrolment);

                    // Log success.
                    $message = get_string(
                        'info:activatedenrolment',
                        'enrol_campusonline',
                        ['userid' => $userid, 'courseid' => $courseid]
                    );
                    locallib::write_log('enrol_user', $message, 0, $courseid, $this->trace, 3);
                }
            }

            // Add role.
            if (!user_has_role_assignment($userid, $roleid, $context->id)) {
                $success = role_assign($roleid, $userid, $context->id);
                $message = get_string(
                    'info:assignedrole',
                    'enrol_campusonline',
                    ['roleid' => $roleid, 'userid' => $userid, 'courseid' => $courseid]
                );
                locallib::write_log('enrol_user', $message, 0, $courseid, $this->trace, 3);
            }

            // Add to assigned roles for later cleanup.
            $assignedroles[$userid][] = $roleid;

            // Add to group members, to process later.
            if ($groups && property_exists($coenrolment, 'courseGroupUid')) {
                $groupmembers[$coenrolment->courseGroupUid][] = $userid;
            }
        }

        // Cleanup - remove roles and suspend empty enrolments.
        $enrolid = $DB->get_record('enrol', ['courseid' => $courseid, 'enrol' => 'campusonline'])->id;
        $moodlecoenrolments = $DB->get_records('user_enrolments', ['enrolid' => $enrolid, 'status' => 0]);

        foreach ($moodlecoenrolments as $moodlecoenrolment) {
            $userid = $moodlecoenrolment->userid;

            // Check if user is enrolled via our own enrolment method.
            if (!$DB->get_record('user_enrolments', ['enrolid' => $enrolid, 'userid' => $userid, 'status' => 0])) {
                continue;
            }

            // Remove roles.
            $roles = get_user_roles($context, $userid, false);
            foreach ($roles as $role) {
                if (!array_key_exists($userid, $assignedroles) || !in_array($role->roleid, $assignedroles[$userid])) {
                    role_unassign($role->roleid, $userid, $context->id);
                    $message = get_string(
                        'info:removedrole',
                        'enrol_campusonline',
                        ['roleid' => $role->roleid, 'userid' => $userid, 'courseid' => $courseid]
                    );
                    locallib::write_log('enrol_user', $message, 0, $courseid, $this->trace, 3);
                }
            }

            // Suspend enrolments with no roles left.
            $roles = get_user_roles($context, $userid, false);
            if (empty($roles)) {
                $enrolment = $DB->get_record('user_enrolments', ['enrolid' => $enrolid, 'userid' => $userid]);
                $enrolment->status = 1;
                $DB->update_record('user_enrolments', $enrolment);
                $message = get_string(
                    'info:suspendedenrolment',
                    'enrol_campusonline',
                    ['userid' => $userid, 'courseid' => $courseid]
                );
                locallib::write_log('enrol_user', $message, 0, $courseid, $this->trace, 3);
            }
        }

        // Create/update groups.
        foreach ($groupmembers as $groupuid => $members) {
            // Get group.
            if ($group = $DB->get_record('groups', ['courseid' => $courseid, 'idnumber' => $groupuid])) {
                $groupid = $group->id;
            } else {
                // Create group.
                $group = new \stdClass();
                $group->courseid = $courseid;
                $group->name = $groups[$groupuid];
                $group->idnumber = $groupuid;
                $group->description = self::CREATED_BY;
                $group->timecreated = time();
                $group->timemodified = time();
                $groupid = groups_create_group($group);

                // Add to grouping.
                if (!$DB->get_record('groupings_groups', ['groupid' => $groupid, 'groupingid' => $groupingid])) {
                    $groupinggroup = new \stdClass();
                    $groupinggroup->groupid = $groupid;
                    $groupinggroup->groupingid = $groupingid;
                    $groupinggroup->timeadded = time();
                    $DB->insert_record('groupings_groups', $groupinggroup);
                }

                $message = get_string(
                    'info:createdmoodlegroup',
                    'enrol_campusonline',
                    ['groupname' => $group->name, 'group_uid' => $groupuid]
                );
                locallib::write_log('sync_groups', $message, 0, $courseid, $this->trace, 3);
            }

            // Add to array with moodle group ids for later cleanup.
            $groupmembersmoodle[$groupid] = $members;

            // Add members.
            foreach ($members as $userid) {
                if (!groups_is_member($groupid, $userid)) {
                    groups_add_member($groupid, $userid);
                    $message = get_string(
                        'info:addedusertogroup',
                        'enrol_campusonline',
                        ['userid' => $userid, 'groupname' => $group->name]
                    );
                    locallib::write_log('sync_groups', $message, 0, $courseid, $this->trace, 3);
                }
            }
        }

        // Remove members.
        $coursegroups = $DB->get_records('groupings_groups', ['groupingid' => $groupingid]);
        foreach ($coursegroups as $coursegroup) {
            $groupid = $coursegroup->groupid;

            $members = groups_get_members($groupid);
            foreach ($members as $member) {
                if (!array_key_exists($groupid, $groupmembersmoodle) || !in_array($member->id, $groupmembersmoodle[$groupid])) {
                    groups_remove_member($groupid, $member->id);
                    $message = get_string(
                        'info:removeduserfromgroup',
                        'enrol_campusonline',
                        ['userid' => $member->id, 'groupname' => $group->name]
                    );
                    locallib::write_log('sync_groups', $message, 0, $courseid, $this->trace, 3);
                }
            }
        }
    }

    /**
     * Syncs selected organisations from CAMPUSonline.
     *
     * @param bool $previewonly if true, only returns sorted org ids without syncing
     *
     * @return array|void
     */
    public function sync_orgs($previewonly = false) {

        // Get all organisations.
        $this->get_org_data();

        // Get selected organisations.
        $orgids = [];
        $endpoint = 'co-brm-core/org/api/selected-organisation-uids';
        $key = $this->config->orgkey;
        $query = ['key' => $key];
        $result = $this->rest_call($endpoint, $query);
        if (property_exists($result, 'items')) {
            foreach ($result->items as $item) {
                if (property_exists($item, 'organisationUids')) {
                    $orgids = $item->organisationUids;
                }
            }
        }

        // Start output.
        $number = count($orgids);
        if (PHP_SAPI == 'cli' || array_key_exists('traceoutput', $_GET)) {
            $this->trace->output(get_string('info:syncingorganisations', 'enrol_campusonline', $number));
        }

        // Sort selected organisations by number of parents that are also selected,
        // so that sync progresses in the correct order.
        $sortedorgids = [];
        foreach ($orgids as $orgid) {
            $parents = 0;
            $org = $this->orgdata[$orgid];
            while (property_exists($org, 'parentUid')) {
                $parentid = $org->parentUid;
                $org = $this->orgdata[$parentid];
                $parents++;
            }
            $sortedorgids[$parents][] = $orgid;
        }
        asort($sortedorgids);

        // If preview only, return sorted org ids.
        if ($previewonly) {
            return [
                'orgdata' => $this->orgdata,
                'sortedorgids' => $sortedorgids,
            ];
        }

        // Sync organisations.
        foreach ($sortedorgids as $orgids) {
            foreach ($orgids as $orgid) {
                $this->sync_org($orgid, $sortedorgids);
            }
        }
    }

    /**
     * Syncs a single organisation and its role assignments.
     *
     * @param string $orgid id of org to be synced
     * @param array $sortedorgids sorted array of org ids
     *
     * @return void
     */
    public function sync_org($orgid, $sortedorgids) {

        global $DB;

        // Get org.
        $org = $this->orgdata[$orgid];
        $orguid = $org->uid;

        // Get parent.
        $parent = $this->config->rootcoursecategory;
        if (property_exists($org, 'parentUid') && in_array($org->parentUid, array_merge(...array_values($sortedorgids)))) {
            $parentid = $org->parentUid;
            if ($parentcat = $DB->get_record('course_categories', ['idnumber' => $parentid])) {
                $parent = $parentcat->id;
            }
        }

        // Get course category.
        if ($category = $DB->get_record('course_categories', ['idnumber' => $orguid])) {
            // Get category object.
            $categoryid = $category->id;
            $cat = \core_course_category::get($categoryid);

            // Check if something needs updating.
            $actions = [];
            $name = locallib::normalize_value($org->name);
            if ($category->name != $name) {
                $cat->update(['name' => $name]);
                $actions[] = "updated";
            }

            // Move category.
            if ($category->parent != $parent) {
                $cat->change_parent($parent);
                $actions[] = "moved";
            }

            if (count($actions) > 0) {
                $actions = implode('&', $actions);
                $actions = ucfirst($actions);
                $message = get_string(
                    'info:updatedcategory',
                    'enrol_campusonline',
                    ['actions' => $actions, 'categoryid' => $category->id, 'org_uid' => $orguid, 'categoryname' => $category->name]
                );
                locallib::write_log('sync_org', $message, 0, null, $this->trace, 3);
            } else {
                $message = get_string(
                    'info:categoryexists',
                    'enrol_campusonline',
                    ['categoryid' => $category->id, 'org_uid' => $orguid, 'categoryname' => $category->name]
                );
                locallib::write_log('sync_org', $message, 0, null, $this->trace, 3);
            }
        } else {
            // Create new course category.
            $data = new \stdClass();
            $data->name = locallib::normalize_value($org->name);
            $data->parent = $parent;
            $data->idnumber = $orguid;
            $data->description = self::CREATED_BY;
            $data->timecreated = time();
            $data->timemodified = time();
            if ($category = \core_course_category::create($data)) {
                $message = get_string(
                    'info:createdcategory',
                    'enrol_campusonline',
                    ['categoryid' => $category->id, 'org_uid' => $orguid, 'categoryname' => $category->name]
                );
                locallib::write_log('sync_org', $message, 0, null, $this->trace, 3);
            } else {
                $message = get_string('error:couldnotcreatecategorysimple', 'enrol_campusonline', $orguid);
                locallib::write_log('sync_org', $message, 2, null, $this->trace, 3);
            }
        }

        // Sync org enrollments.
        if ($this->config->syncorgroles && $this->orgroles) {
            $this->sync_org_enrolments($orguid, $category->id);
        }
    }

    /**
     * Syncs role assignments for a single organisation.
     *
     * @param string $orguid
     * @param string $categoryid
     *
     * @return void
     */
    public function sync_org_enrolments($orguid, $categoryid) {

        global $DB;

        foreach ($this->orgroles as $roleid => $rolename) {
            $userids = [];

            // Convert moodle rolename to CO rolename.
            $corole = strtoupper(substr($rolename, 3));

            // Get role assignments from CO.
            $endpoint = 'co-auth/auth/api/person-uids';
            $query = [
                'role_name' => $rolename,
                'context' => "org-$orguid",
            ];
            $result = $this->rest_call($endpoint, $query);
            if ($result = $this->rest_call($endpoint, $query)) {
                if (property_exists($result, 'items')) {
                    foreach ($result->items as $uid) {
                        if ($userid = $this->get_moodle_user_id($uid)) {
                            $userids[] = $userid;
                        } else if ($this->config->enrolsynccreateusers) {
                            if ($userid = $this->create_moodle_user($uid)) {
                                $userids[] = $userid;
                            }
                        }
                    }
                }
            }

            // Remove Moodle roles.
            $context = \context_coursecat::instance($categoryid);
            $users = get_role_users($roleid, $context);
            foreach ($users as $user) {
                if (!in_array($user->id, $userids)) {
                    role_unassign($roleid, $user->id, $context->id);
                    $message = get_string(
                        'info:removedrolefromuser',
                        'enrol_campusonline',
                        ['roleid' => $roleid, 'userid' => $user->id, 'org_uid' => $orguid]
                    );
                    locallib::write_log('sync_org_roles', $message, 0, null, $this->trace, 5);
                }
            }

            // Assign Moodle roles.
            foreach ($userids as $userid) {
                if (!user_has_role_assignment($userid, $roleid, $context->id)) {
                    $success = role_assign($roleid, $userid, $context->id);
                    $message = get_string(
                        'info:assignedroletouser',
                        'enrol_campusonline',
                        ['roleid' => $roleid, 'userid' => $userid, 'org_uid' => $orguid]
                    );
                    locallib::write_log('sync_org_roles', $message, 0, null, $this->trace, 5);
                }
            }
        }
    }

    /**
     * Syncs users.
     *
     * @return void
     */
    public function sync_users() {

        $persons = $this->get_persons();

        foreach ($persons as $person) {
            // Get Moodle user.
            $uid = $person['uid'];
            $userid = $this->get_moodle_user_id($uid);

            // Create new user if needed & allowed.
            if (!$userid) {
                if ($this->config->enrolsynccreateusers) {
                    if (!$userid = $this->create_moodle_user($uid)) {
                        continue;
                    }
                } else {
                    // Log warning.
                    $message = get_string('warning:skippedsyncinguserdata', 'enrol_campusonline', $uid);
                    locallib::write_log('sync_user', $message, 1, null, $this->trace, 3);
                    continue;
                }
            }

            // Load Moodle User.
            $user = \core_user::get_user($userid);
            $this->update_moodle_user($user, $person);
        }
    }

    /**
     * Updates a single Moodle users's data.
     *
     * @param object $user
     * @param array $person
     *
     * @return void
     */
    public function update_moodle_user($user, $person) {

        $uid = $person['uid'];
        $needsupdate = false;
        foreach (locallib::USER_FIELDS as $field => $default) {
            // Only update username if allowed.
            if ($field == 'username' && !$this->config->user_allowusernameupdate) {
                continue;
            }

            // Only update email if allowed.
            if ($field == 'email' && !$this->config->user_allowemailupdate) {
                continue;
            }

            $value = locallib::get_field_value("user_$field", $person);

            // Sanitize usernames.
            if ($field == 'username') {
                $value = strtolower($value);
            }

            if ($user->$field != $value) {
                $user->$field = $value;
                $needsupdate = true;
            }
        }

        if ($needsupdate) {
            user_update_user($user, false);
        }

        // Load profile data into user object.
        $needsupdatep = locallib::set_custom_user_fields($user, $person);

        // Log update.
        if ($needsupdate || $needsupdatep) {
            $message = get_string('info:updatedmoodleuser', 'enrol_campusonline', ['userid' => $user->id, 'uid' => $uid]);
            locallib::write_log('sync_user', $message, 0, null, $this->trace, 3);
        } else {
            // Write trace output that no update is necessary.
            if (array_key_exists('traceoutput', $_GET)) {
                $this->trace->output(
                    get_string(
                        'info:noupdatenecessary',
                        'enrol_campusonline',
                        ['userid' => $user->id, 'uid' => $uid]
                    )
                );
            }
        }
    }

    /**
     * Creates a new Moodle user.
     *
     * @param string $uid
     * @param string|null $usertype
     *
     * @return int|null $userid
     */
    private function create_moodle_user($uid, $usertype = null): int|null {

        global $CFG, $DB;

        // Get full person data from CAMPUSonline.
        $userdata = $this->get_person_data($uid);

        // When separate users for students and staff are enabled in config, prefix the person uid.
        if ($this->config->separateusers && $usertype) {
            $uid = $usertype . '_' . $uid;
            $userdata['uid'] = $uid;
        }

        // Build user.
        $user = locallib::build_user($userdata);

        // Convert from array to object.
        $user = (object)$user;

        // Set required fields.
        $user->mnethostid = $CFG->mnet_localhost_id;
        $user->confirmed = 1;

        // Check if all required fields are set.
        foreach (locallib::USER_FIELDS_NOEMPTY as $check) {
            if ($user->$check == '') {
                $message = get_string(
                    'error:couldnotcreateuser',
                    'enrol_campusonline',
                    ['uid' => $uid, 'check' => $check]
                );
                locallib::write_log('create_user', $message, 2, null, $this->trace, 3);
                return null;
            }
        }

        // Check if all unique fields are indeed unique.
        foreach (locallib::USER_FIELDS_UNIQUE as $check) {
            $value = $user->$check;
            if ($duplicate = $DB->get_record('user', [$check => $value])) {
                // Log error and return null.
                $message = get_string(
                    'error:couldnotcreateuserunique',
                    'enrol_campusonline',
                    ['uid' => $uid, 'check' => $check, 'value' => $value, 'duplicate' => $duplicate->id]
                );
                locallib::write_log('create_user', $message, 2, null, $this->trace, 3);
                return null;
            }
        }

        // Create user.
        if ($userid = user_create_user($user)) {
            $message = get_string('info:createdmoodleuser', 'enrol_campusonline', ['userid' => $userid, 'uid' => $uid]);
            $status = 0;
        } else {
            $message = get_string('error:couldnotcreateuser', 'enrol_campusonline', $uid);
            $status = 2;
        }
        locallib::write_log('create_user', $message, $status, null, $this->trace, 3);

        // Update custom fields.
        $user = \core_user::get_user($userid);
        locallib::set_custom_user_fields($user, $userdata);

        return $userid;
    }

    /**
     * Enriches courses with additional data from other endpoints.
     *
     * @param array $courses
     *
     * @return array $courses
     */
    private function enrich_courses($courses): array {

        $this->get_org_data();
        $this->get_semester_data();

        $enrichedcourses = [];
        foreach ($courses as $course) {
            $sanitizedcourse = [];
            $course = (array)$course;

            // Values from course endpoint.
            foreach ($course as $key => $value) {
                $value = locallib::normalize_value($value);
                $sanitizedcourse["course:$key"] = $value;
            }

            // Attach values from org endpoint.
            $org = $this->orgdata[$course['organisationUid']];
            $org = (array)$org;
            foreach ($org as $key => $value) {
                $value = locallib::normalize_value($value);
                $sanitizedcourse["org:$key"] = $value;
            }

            // Attach values from semester endpoint.
            $semester = $this->semesterdata[$course['semesterKey']] ?? null;
            $semester = (array)$semester;
            foreach ($semester as $key => $value) {
                $value = locallib::normalize_value($value);
                $sanitizedcourse["semester:$key"] = $value;

                // Add short semester key because some sites might find it useful.
                if ($key == 'key') {
                    $shortkey = substr($value, -3);
                    $sanitizedcourse['semester:key:short'] = $shortkey;
                }
            }

            // Get values from course description endpoint.
            if ($this->config->getcoursedescription) {
                $descriptions = $this->get_course_descriptions($course['uid']);
                foreach ($descriptions as $itemkey => $content) {
                    $content = (array)$content;
                    if (!array_key_exists('value', $content)) {
                        continue;
                    }
                    $values = $content['value'];
                    foreach ($values as $lang => $value) {
                        $value = locallib::normalize_value($value);
                        $key = "description:$itemkey:$lang";
                        $sanitizedcourse[$key] = $value;
                    }
                }
            }

            $enrichedcourses[] = $sanitizedcourse;
        }

        return $enrichedcourses;
    }

    /**
     * Gets org data from CAMPUSonline.
     *
     * @return void
     */
    private function get_org_data(): void {

        if ($this->orgdata) {
            return;
        }

        $endpoint = 'co-brm-core/org/api/organisations';
        $result = $this->rest_call($endpoint, null, 'GET', true);

        // Add to class property.
        $this->orgdata = [];
        if (property_exists($result, 'items')) {
            foreach ($result->items as $item) {
                $this->orgdata[$item->uid] = $item;
            }
        }
    }

    /**
     * Gets semester data from CAMPUSonline.
     */
    private function get_semester_data(): void {

        if ($this->semesterdata) {
            return;
        }

        $endpoint = 'co-sm-core/semester/api/semesters';
        $result = $this->rest_call($endpoint, null, 'GET', true);

        // Add to class property.
        $this->semesterdata = [];
        if (property_exists($result, 'items')) {
            foreach ($result->items as $item) {
                $this->semesterdata[$item->key] = $item;
            }
        }
    }

    /**
     * Calls REST API with pagination.
     *
     * @param string $endpoint The API endpoint to call.
     * @param array $query Query parameters to pass to the API.
     * @param string $method HTTP method (GET by default).
     * @param string $alwayspage Whether to always page through the API.
     *
     * @return object All collected items from paginated API responses.
     */
    public function rest_call($endpoint, $query = null, $method = 'GET', $alwayspage = false) {

        // Set base URL and initialize the HTTP client.
        $url = $this->config->endpoint . '/' . $endpoint;
        $client = new Client([
            'base_uri' => $url,
            'timeout' => $this->timeout,
            'connect_timeout' => $this->connecttimeout,
        ]);

        // Set payload key.
        if ($method == 'GET') {
            $payload = 'query';
        } else {
            $payload = 'json';
        }

        // Initialize variables.
        $allitems = [];
        $cursor = null;
        $responseobject = new \stdClass();
        $page = !array_key_exists('limit', $_GET) || $alwayspage;

        do {
            // Update the query with the cursor, if available.
            if ($cursor !== null) {
                $query['cursor'] = $cursor;

                // Debug message.
                if ($this->config->restcalls && PHP_SAPI === 'cli') {
                    $count = count($allitems);
                    $this->trace->output(
                        get_string(
                            'info:pagingcursor',
                            'enrol_campusonline',
                            ['cursor' => $cursor, 'count' => $count]
                        )
                    );
                }
            }

            // Make the API request.
            $retry = 0; // Track retries.
            $retries = $this->config->maxretries;
            do {
                // Update the token if necessary.
                if ($this->maxtokenage < time() - $this->tokencreation) {
                    $this->update_token();
                }

                try {
                    // Debug message.
                    if ($this->config->restcalls && PHP_SAPI === 'cli') {
                        if ($query) {
                            $jsonquery = json_encode($query);
                        } else {
                            $jsonquery = '';
                        }
                        $this->trace->output(
                            get_string(
                                'info:apirequest',
                                'enrol_campusonline',
                                ['method' => $method, 'endpoint' => $endpoint, 'query' => $jsonquery]
                            )
                        );
                    }

                    $response = $client->request($method, $url, [
                        'headers' => [
                            'accept' => 'application/json',
                            'Content-Type' => 'application/json',
                            'Authorization' => 'Bearer ' . $this->token,
                        ],
                        $payload => $query,
                    ]);

                    // Decode the response.
                    $responsebody = $response->getBody()->getContents();
                    $responseobject = json_decode($responsebody, false);

                    // Merge the current page's items with the collected items.
                    if (property_exists($responseobject, 'items')) {
                        $allitems = array_merge($allitems, $responseobject->items);
                    }

                    // Check if there is a next cursor for pagination.
                    if (property_exists($responseobject, 'nextCursor')) {
                        $cursor = $responseobject->nextCursor;
                    } else {
                        $cursor = null;
                    }

                    // Determine if we need to keep paging.
                    $page = !array_key_exists('limit', $_GET) || $alwayspage;

                    // Exit the loop on success.
                    break;

                    // Error handling.
                } catch (\Throwable $e) {
                    $retry++;
                    if ($retry > $retries) {
                        $error = $e->getMessage();
                        $message = get_string('error:requestexception', 'enrol_campusonline', $error);
                        locallib::write_log('error', $message, 2, null, $this->trace);

                        // Create object with exception message.
                        $responseobject = new \stdClass();
                        $responseobject->exception = $error;
                        return $responseobject;
                    } else {
                        locallib::write_log(
                            'warning',
                            get_string(
                                'warning:retryingfailedrequest',
                                'enrol_campusonline',
                                $e->getMessage()
                            ),
                            1,
                            null,
                            $this->trace
                        );
                        sleep(3);
                    }
                }
            } while (true);
        } while ($page && $cursor !== null);

        // Return the complete collection of items.
        if ($allitems) {
            return (object)[
                'items' => $allitems,
            ];
        }
        return ($responseobject);
    }

    /**
     * Sets the moodle course URL in CAMPUSonline.
     *
     * @param object $course
     * @param string $groupuid
     *
     * @return void
     */
    private function set_moodle_course_url($course, $groupuid = null) {

        global $DB;

        // Add own uid.
        $courseuid = explode(':', $course->idnumber)[0];
        $courseuids[] = $courseuid;

        // Add other uids.
        $handler = \core_customfield\handler::get_handler('core_course', 'course');
        $datas = $handler->get_instance_data($course->id);
        $metadata = [];
        foreach ($datas as $data) {
            if (empty($data->get_value())) {
                continue;
            }
            $cat = $data->get_field()->get_category()->get('name');
            $metadata[$data->get_field()->get('shortname')] = $cat . ': ' . $data->get_value();
        }

        // Get course custom field directly via DB, to avoid having to deal with custom field API.
        $otheruids = $DB->get_field(
            'customfield_data',
            'charvalue',
            ['instanceid' => $course->id, 'fieldid' => $this->customfieldids['campusonline_other_co_course_uids']]
        );
        if ($otheruids) {
            $otheruids = preg_split('/\s*,\s*/', $otheruids);
            $courseuids = array_merge($courseuids, $otheruids);
        }

        // Create URL.
        $endpoint = 'co-tm-core/course/api/e-learning-infos';
        $moodleurl = new moodle_url('/course/view.php', ['id' => $course->id]);
        $url = $moodleurl->__toString();

        $query['externalUrl'] = $url;
        if ($groupuid) {
            $query['courseGroupUid'] = $groupuid;
        }

        // Set Moodle course URL in CAMPUSonline.
        foreach ($courseuids as $courseuid) {
            $query['courseUid'] = $courseuid;

            // Log success.
            if ($result = $this->rest_call($endpoint, $query, 'POST')) {
                if (property_exists($result, 'externalUrl')) {
                    $url = $result->externalUrl;
                    $message = get_string(
                        'info:updatedcampuscourse',
                        'enrol_campusonline',
                        ['course_uid' => $courseuid, 'url' => $url]
                    );
                    locallib::write_log('update_course', $message, 0, $course->id, $this->trace, 3);
                    continue;
                }

                // Error.
                $message = get_string(
                    'error:couldnotupdatecampuscourse',
                    'enrol_campusonline',
                    ['course_uid' => $courseuid, 'url' => $url]
                );
                locallib::write_log('update_course', $message, 2, $course->id, $this->trace, 3);
            }
        }
    }

    /**
     * Gets access token for REST calls.
     *
     * @return void
     */
    private function update_token(): void {

        $path = $this->config->endpoint;
        $clientid = $this->config->clientid;
        $secret = $this->config->clientsecret;

        if (!$path || !$clientid || !$secret) {
            $this->error = get_string('error:config', 'enrol_campusonline');
            return;
        }

        $path = rtrim($path, '/');
        $url = $path . '/public/sec/auth/realms/CAMPUSonline_SP/protocol/openid-connect/token';
        $client = new Client([
            'base_uri' => $url,
            'timeout' => (float) $this->config->timeout,
            'connect_timeout' => (float) $this->config->connect_timeout,
        ]);

        try {
            $response = $client->request('POST', $url, [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded',
                ],
                'form_params' => [
                    'grant_type' => 'client_credentials',
                    'client_id' => $clientid,
                    'client_secret' => $secret,
                ],
            ]);

            // Analyze response.
            $responsebody = $response->getBody()->getContents();
            $responseobject = json_decode($responsebody, false);
            $responsearray = (array)$responseobject;

            // Analyze response.
            if (array_key_exists('error', $responsearray)) {
                $this->error = $responsearray['error'] . ': ' . $responsearray['error_description'];
            } else if (array_key_exists('access_token', $responsearray)) {
                $this->token = $responsearray['access_token'];
            } else {
                $this->error = $responsearray['error'] . ': ' . get_string('error:unknown', 'enrol_campusonline');
            }

            // Save creation time of the token.
            $this->tokencreation = time();

            // Error handling.
        } catch (\Throwable $e) {
            $error = $e->getMessage();
            $this->error = $error;
            $message = get_string('error:requestexception', 'enrol_campusonline', $error);
            locallib::write_log('update_token', $message, 2, null, $this->trace);
        }
    }
}
