// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * AMD module defining course filter buttons behaviour.
 */
define([], function() {

    /**
     * Setup behaviour of course filter button.
     * @param {Element} control Button.
     * @param {Element} coursesList Courses list HTML element.
     * @param {Number[]} coursesInFilter List of course ids that are filtered in by this filter.
     * @param {Number[]} categoriesWithCoursesInFilter List of course category ids that contain courses that are filtered in.
     */
    function setupFilterButton(control, coursesList, coursesInFilter, categoriesWithCoursesInFilter) {
        // Define control behavior.
        var toggleCoursesVisibility = function() {
            var hide = control.classList.contains('active');
            coursesList.querySelectorAll('[data-courseid]').forEach(function(course) {
                if (!coursesInFilter.includes(Number(course.dataset.courseid))) {
                    course.hidden = hide;
                    course.ariaHidden = hide;
                }
            });
            coursesList.querySelectorAll('[data-categoryid]').forEach(function(category) {
                if (!categoriesWithCoursesInFilter.includes(Number(category.dataset.categoryid))) {
                    category.hidden = hide;
                    category.ariaHidden = hide;
                }
            });
        };
        control.addEventListener('click', function() {
            // If another filter was active, toggle it off.
            document.querySelectorAll('[data-role=local_coursefilters-filter-button].active').forEach(function(filterButton) {
                if (filterButton.id != control.id) { // Do not click on current filter!
                    filterButton.click();
                }
            });
            // Then toggle this filter.
            control.classList.toggle('btn-secondary');
            control.classList.toggle('btn-primary');
            control.classList.toggle('active');
            control.ariaPressed = control.classList.contains('active');
            toggleCoursesVisibility();
        });

        // Manage dynamically loaded course categories.
        // This will automatically apply this filter, if active, on newly loaded categories.
        var delayUntilNotEmpty = function(div, func) {
            setTimeout(function() {
                if (div.childNodes.length) {
                    func();
                } else {
                    delayUntilNotEmpty(div, func);
                }
            }, 50);
        };
        var toggleCoursesVisibilityInNewlyLoadedCategory = function(category) {
            delayUntilNotEmpty(category.querySelector('.content'), function() {
                if (control.classList.contains('active')) {
                    toggleCoursesVisibility();
                }
                category.querySelectorAll('.subcategories').forEach(function(list) {
                    manageDynamicLoad(list);
                });
            });
        };
        var manageDynamicLoad = function(list) {
            list.querySelectorAll('.category.notloaded').forEach(function(category) {
                category.addEventListener('click', function() {
                    toggleCoursesVisibilityInNewlyLoadedCategory(category);
                });
            });
            list.querySelectorAll('.collapseexpand').forEach(function(collapseexpand) {
                collapseexpand.addEventListener('click', function() {
                    if (!collapseexpand.classList.contains('collapse-all')) {
                        list.querySelectorAll('.category.notloaded').forEach(function(category) {
                            toggleCoursesVisibilityInNewlyLoadedCategory(category);
                        });
                    }
                });
            });
        };
        manageDynamicLoad(coursesList);
    }

    return {
        init: function(filterLabel, coursesInFilter, categoriesWithCoursesInFilter, disabled) {

            // Create the control.
            var control = document.createElement('button');
            control.id = Date.now().toString(36) + Math.random().toString(36).substring(2); // Unique ID.
            control.classList.add('btn', 'btn-secondary', 'float-right', 'mb-1', 'mx-1');
            control.dataset.role = 'local_coursefilters-filter-button';
            control.type = 'button';
            control.innerHTML = filterLabel;
            control.ariaPressed = false;

            // Find the courses list to filter.
            var coursesList = document.querySelector('.course_category_tree, .course-search-result');

            if (disabled) {
                control.disabled = true;
            } else {
                setupFilterButton(control, coursesList, coursesInFilter, categoriesWithCoursesInFilter);
            }

            // Add the control to the page.
            var div = document.createElement('div');
            div.classList.add('clearfix');
            div.appendChild(control);
            coursesList.parentElement.insertBefore(div, coursesList);
        }
    };
});
