<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace local_coursefilters;

/**
 * Base class for all course filters.
 * @package    local_coursefilters
 * @copyright  2024 Astor Bizard
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
abstract class filter {
    /**
     * Whether this filter requires current user to be authenticated to make sense and be displayed.
     * @return bool
     */
    abstract public function needs_auth(): bool;

    /**
     * Whether this filter should be displayed even if no course is being filtered in.
     * @return bool
     */
    abstract public function show_if_none(): bool;

    /**
     * Compute if given course is filtered in.
     * @param int $courseid Course ID.
     * @return bool
     */
    abstract public function course_is_in_filter($courseid): bool;

    /**
     * This filter name. Defaults to class name without trailing "_filter".
     * @return string
     */
    protected function get_name() {
        $class = get_class($this);
        return substr($class, (strrpos($class, "\\") + 1) ?: 0, -7);
    }

    /**
     * Get this filter label that can be displayed on corresponding button.
     * @return string Localized label.
     */
    public function get_label() {
        return get_string('filterby' . $this->get_name(), 'local_coursefilters');
    }

    /**
     * Get decorative icon for this filter button.
     * @param string $classes Optional additionnal classes to add to the icon.
     * @return string HTML fragment for icon.
     */
    public function get_icon($classes = '') {
        global $OUTPUT;
        // Icon is decorative only, $OUTPUT->pix_icon does handle accessibility when not providing alt text.
        return $OUTPUT->pix_icon($this->get_name(), '', 'local_coursefilters', [ 'class' => $classes ]);
    }
}
