<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Callbacks used by Moodle API.
 * @package    local_coursefilters
 * @copyright  2024 Astor Bizard
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * Definition of Fontawesome icons mapping.
 * @return string[] Fontawesome icons mapping.
 */
function local_coursefilters_get_fontawesome_icon_map() {
    return [
            'local_coursefilters:openaccess' => 'fa-sign-in',
            'local_coursefilters:enrolled' => 'fa-user',
            'local_coursefilters:notenrolled' => 'fa-user-times',
            'local_coursefilters:teacher' => 'fa-graduation-cap',
    ];
}

/**
 * Hook called before displaying page footer.
 */
function local_coursefilters_before_footer() {
    global $PAGE, $DB;

    foreach ([ 'openaccess', 'enrolled', 'notenrolled', 'teacher' ] as $filtername) {
        if (!in_array($PAGE->pagetype, explode(',', get_config('local_coursefilters', $filtername . 'filterlocations')))) {
            // Not on a page where the filter should be present.
            continue;
        }
        $filter = local_coursefilters\filter_factory::get_filter($filtername);
        if ($filter->needs_auth() && (!isloggedin() || isguestuser())) { // Login check.
            continue;
        }
        // On course search, we don't know which courses will show up so no count.
        $shownumber = $PAGE->pagetype != 'course-search';
        $categoryfilter = $PAGE->pagetype == 'course-index-category' ? optional_param('categoryid', null, PARAM_INT) : null;
        $courses = [];
        $categorieswithcourses = [];
        $isempty = true;
        // Loop over every course (no need to filter out site course as it is in no category).
        $sql = "SELECT c.id, cat.path
                  FROM {course} c
                  JOIN {course_categories} cat ON cat.id = c.category";
        foreach ($DB->get_records_sql($sql) as $course) {
            if ($filter->course_is_in_filter($course->id)) {
                // At least one course exists in this filter. It may not be displayed (e.g. if viewing another category),
                // but consider the filter non-empty (it means "it makes sense to the current user that this filter exists").
                $isempty = false;

                $coursecategories = explode('/', $course->path);
                if ($categoryfilter !== null && !in_array($categoryfilter, $coursecategories)) {
                    // This course is not in the category user is viewing.
                    continue;
                }
                $courses[] = (int) $course->id;
                foreach ($coursecategories as $category) {
                    if (!empty($category)) {
                        $categorieswithcourses[(int) $category] = true;
                    }
                }
            }
        }
        if (!$isempty || $filter->show_if_none()) {
            // Icon is decorative only, $OUTPUT->pix_icon does handle accessibility when not providing alt text.
            $label = $filter->get_label() . ($shownumber ? ' (' . count($courses) . ')' : '') . $filter->get_icon('mx-1');
            $PAGE->requires->js_call_amd('local_coursefilters/courses_filter', 'init',
                    [ $label, $courses, array_keys($categorieswithcourses), $shownumber && count($courses) == 0 ]);
        }
    }
}
