<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace local_coursefilters;

/**
 * Course filter for courses where current user is a teacher.
 * @package    local_coursefilters
 * @copyright  2024 Astor Bizard
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class teacher_filter extends filter {
    /**
     * @var array Array of courses where current user is a teacher.
     */
    protected $teachercourses;

    /**
     * Constructor.
     */
    public function __construct() {
        global $DB, $USER;
        $teacherroles = explode(',', get_config('local_coursefilters', 'teacherfilterroles'));
        list($rolesinsql, $params) = $DB->get_in_or_equal($teacherroles, SQL_PARAMS_NAMED);
        $sql = "SELECT c.id
                  FROM {course} c
                  JOIN {context} ctx ON ctx.instanceid = c.id AND ctx.contextlevel = :courselevel
                  JOIN {role_assignments} ra ON ra.contextid = ctx.id
                 WHERE ra.userid = :userid AND ra.roleid $rolesinsql";
        $params['userid'] = $USER->id;
        $params['courselevel'] = CONTEXT_COURSE;
        $this->teachercourses = $DB->get_records_sql($sql, $params);
    }

    /**
     * {@inheritDoc}
     * @see \local_coursefilters\filter::needs_auth()
     */
    public function needs_auth(): bool {
        return true;
    }

    /**
     * {@inheritDoc}
     * @see \local_coursefilters\filter::show_if_none()
     */
    public function show_if_none(): bool {
        return false;
    }

    /**
     * {@inheritDoc}
     * @param int $courseid Course ID.
     * @see \local_coursefilters\filter::course_is_in_filter()
     */
    public function course_is_in_filter($courseid): bool {
        return isset($this->teachercourses[$courseid]);
    }
}
