<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * No file description
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
namespace local_treestudyplan\form;
defined('MOODLE_INTERNAL') || die();
require_once($CFG->dirroot."/lib/formslib.php");

/**
 * Base class for moodle forms used in this plugin.
 */
abstract class formbase extends \moodleform {

    /**
     * Create the form while providing the correct defaults for our type of forms
     * @param object $params Custom parameters this form needs
     * @param array $ajaxformdata Provide submitted form data through Ajax here
     * @throws \moodle_exception if parameters are invalid or access is denied for a specific reason.
     */
    public function __construct($params, $ajaxformdata=null) {
        $customdata = static::init_customdata($params);
        if (static::check_security($customdata) ) {
            parent::__construct(null, (array)$customdata, 'post', '', null, true, $ajaxformdata);
            // Initialize the initial data based on the parameter validation.
            $this->set_data($this->init_formdata((object)$this->_customdata));
        } else {
            throw new \moodle_exception('accessexception', 'core');
        }
    }

    /**
     * Generate custom data from parameters
     * Also validate parameters and access permissions here
     *
     * @param object $params The parameters for form initialization
     * @return object Form data based on parameters
     */
    abstract public static function init_customdata(object $params);

    /**
     * Generate form data from parameters
     * Also validate parameters and access permissions here
     *
     * @param object $customdata The form customdata built on form initialization in self::init_customdata(...)
     * @return object Form data based on parameters
     */
    abstract public function init_formdata(object $customdata);

    /**
     * Validate security access for this form based on the provided parameters
     * Return true if validation passes, false or throw an exception if it does not.
     *
     * @param object $customdata The form customdata built on form initialization
     * @return bool True if security validation passes.
     * @throws \moodle_exception if access denied for a specific reason in self::init_customdata(...)
     */
    public static function check_security(object $customdata) {
        return true;
    }

    /**
     * Process the submission and perform necessary actions
     * @param object $entry The processed form data
     * @return mixed Data to pass to receiver if submission successful
     * @throws \moodle_exception if an error must be given for a specific reason.
     */
    abstract protected function process_submitted_data(object $entry);

    /**
     * Process the submission and perform necessary actions
     * @return mixed Data to pass to receiver if submission successful
     * @throws \moodle_exception if an error must be given for a specific reason.
     */
    public function process_submission() {
        $data = $this->get_data();

        if ($data) {
            return $this->process_submitted_data($data);
        } else {
            throw new \moodle_exception('no_form_data', 'local_treestudyplan', '', null, $data);
        }
    }
}
