<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Class to manage connection lines between study items
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan;
use core_external\external_api;
use core_external\external_function_parameters;
use core_external\external_description;
use core_external\external_value;
use core_external\external_multiple_structure;
use core_external\external_single_structure;
use core\context;
use core\context\system as context_system;
use core\context\course as context_course;
use core\context\coursecat as context_coursecat;

defined('MOODLE_INTERNAL') || die();

require_once($CFG->libdir.'/externallib.php');

/**
 * Class to manage connection lines between study items
 * * @SuppressWarnings(PHPMD.TooManyPublicMethods)
 */
class studyitemconnection {

    /**
     * Database table for this model
     * @var string */
    const TABLE = "local_treestudyplan_connect";

    /**
     * Holds database record
     * @var stdClass
     */
    private $r;
    /** @var int */
    private $id;

    /**
     * Construct new object based on database object
     * @param stdClass $r Database record
     */
    protected function __construct($r) {
        $this->r = $r;
        $this->id = $r->id;
    }

    /**
     * Webservice structure for basic info
     * @param int $value Webservice requirement constant
     */
    public static function structure($value = VALUE_REQUIRED): external_description {
        return new external_single_structure([
            'id' => new external_value(PARAM_INT, 'id of connection'),
            'from_id' => new external_value(PARAM_INT, 'id of start item'),
            'to_id' => new external_value(PARAM_INT, 'id of end item'),
        ], '', $value);
    }

    /**
     * Webservice model for basic info
     * @return array Webservice data model
     */
    public function model() {
        return ['id' => $this->r->id, 'from_id' => $this->r->from_id, 'to_id' => $this->r->to_id];
    }

    /**
     * Get originating studyitem for this connection
     */
    public function from_item(): studyitem {
        return studyitem::find_by_id($this->r->from_id);
    }

    /**
     * Get target studyitem for this connection
     */
    public function to_item(): studyitem {
        return studyitem::find_by_id($this->r->to_id);
    }

    /**
     * Get originating studyitem id for this connection
     */
    public function from_id(): int {
        return $this->r->from_id;
    }

    /**
     * Get target studyitem id for this connection
     */
    public function to_id(): int {
        return $this->r->to_id;
    }

    /**
     * Get all connections originating at a given studyitem's id
     * @param int $itemid Id of the studyitem
     * @return studyitemconnection[]
     */
    public static function find_outgoing($itemid): array {
        global $DB;
        $list = [];
        $connout = $DB->get_records(self::TABLE, ['from_id' => $itemid]);
        foreach ($connout as $c) {
            $list[] = new self($c);
        }
        return $list;
    }

    /**
     * Get all connections terminating at a given studyitem's id
     * @param int $itemid Id of the studyitem
     * @return studyitemconnection[]
     */
    public static function find_incoming($itemid) {
        global $DB;
        $list = [];
        $connin = $DB->get_records(self::TABLE, ['to_id' => $itemid]);
        foreach ($connin as $c) {
            $list[] = new self($c);
        }
        return $list;
    }

    /**
     * Get all connections terminating at a given studyitem's id
     * @param int $itemid Id of the studyitem
     * @return array [$incoming, $outgoing]
     */
    public static function find_all($itemid) {
        global $DB;
        $incoming = [];
        $outgoing = [];
        $conns = $DB->get_records_sql("SELECT * from {local_treestudyplan_connect} WHERE to_id = :to_id OR from_id = :from_id",
                                      ['to_id' => $itemid, 'from_id' => $itemid]);
        foreach ($conns as $c) {
            $conn = new self($c);
            if ($c->from_id == $itemid) {
                $outgoing[] = $conn;
            }
            if ($c->to_id == $itemid) {
                $incoming[] = $conn;
            }
        }
        return [$incoming, $outgoing];
    }


    /**
     * Create a new connection between two study items
     * @param int $fromid Id of the originating studyitem
     * @param int $toid Id of the target studyitem
     */
    public static function connect($fromid, $toid) {
        global $DB;

        // Check if link already exists.

        if (!$DB->record_exists(self::TABLE, ['from_id' => $fromid, 'to_id' => $toid])) {
            $id = $DB->insert_record(self::TABLE, [
                'from_id' => $fromid,
                'to_id' => $toid,
            ]);

            return new self($DB->get_record(self::TABLE, ['id' => $id]));

        } else {
            return new self($DB->get_record(self::TABLE, ['from_id' => $fromid, 'to_id' => $toid]));
        }
    }

    /**
     * Delete a connection between two study items
     * @param int $fromid Id of the originating studyitem
     * @param int $toid Id of the target studyitem
     */
    public static function disconnect($fromid, $toid) {
        global $DB;

        if ($DB->record_exists(self::TABLE, ['from_id' => $fromid, 'to_id' => $toid])) {
            $DB->delete_records(self::TABLE, [
                'from_id' => $fromid,
                'to_id' => $toid,
            ]);

            return success::success(['msg' => 'Items Disconnected']);
        } else {
            return success::success(['msg' => 'Connection does not exist']);
        }
    }

    /**
     * Clear all connection to and from a given studyitem
     * @param int $id Id of the studyitem
     */
    public static function clear($id) {
        global $DB;
        $DB->delete_records(self::TABLE, ['from_id' => $id]);
        $DB->delete_records(self::TABLE, ['to_id' => $id]);
    }

}
