<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Task to autosync studyplan associated cohorts and users with the courses in the studyplan
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan\task;
defined('MOODLE_INTERNAL') || die();

require_once($CFG->dirroot.'/course/externallib.php');
use local_treestudyplan\studyplan;
use local_treestudyplan\studyline;
use local_treestudyplan\cascadecohortsync;
use local_treestudyplan\cascadeusersync;

/**
 * Task to enrol users and cohorts associated with a studyplan in all courses linked in that studyplan
 */
class autocohortsync extends \core\task\scheduled_task {

    /**
     * Return the task's name as shown in admin screens.
     *
     * @return string
     */
    public function get_name() {
        return get_string('autocohortsync_name', 'local_treestudyplan');
    }

    /**
     * Execute the task.
     */
    public function execute() {
        if (get_config("local_treestudyplan", "csync_enable")) {
            \mtrace("Automatic csync cascading enabled");
            $studyplans = studyplan::find_all();

            foreach ($studyplans as $studyplan) {
                // Only process studyplans that have been marked for change because .
                // A cohort change has occurred or a course has been added....
                if ($studyplan->has_csync_changed()) {
                    \mtrace("Studyplan {$studyplan->shortname()} needs processing");
                    $enroller = new cascadecohortsync($studyplan);
                    $enroller->sync();
                    if (get_config("local_treestudyplan", "csync_users")) {
                        $userenroller = new cascadeusersync($studyplan);
                        $userenroller->sync();
                    }
                    $studyplan->clear_csync_changed();

                }
            }
            \mtrace("Task done");
        } else {
            \mtrace("Automatic csync cascading disabled");
        }
    }

    /**
     * Perform immediate syncronization on a single studyplan.
     * @param studyplan $studyplan Relevant study plan
     */
    public static function syncplan(studyplan $studyplan) {
        $enroller = new cascadecohortsync($studyplan);
        $enroller->sync();

        if (get_config("local_treestudyplan", "csync_users")) {
            $userenroller = new cascadeusersync($studyplan);
            $userenroller->sync();
        }
        $studyplan->clear_csync_changed(); // Clear the csync required flag.
    }

    /**
     * Perform immediate syncronization on a single studyline.
     * @param studyline $line Relevant study line
     */
    public static function syncline(studyline $line) {
        $plan = $line->studyplan();
        $enroller = new cascadecohortsync($plan);
        $enroller->sync($line);

        if (get_config("local_treestudyplan", "csync_users")) {
            $userenroller = new cascadeusersync($plan);
            $userenroller->sync($line);
        }
        // Leave the csync required flag for the next auto update.
    }

}
