<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Scan submissions in assign activities
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan\local\ungradedscanners;

/**
 * Activity scanner for assignment activity
 */
class assign_scanner extends scanner_base {

    /**
     * Retrieve the ungraded submissions in this activity
     */
    protected function get_ungraded_submissions() {
        global $DB;
        $sql = "SELECT DISTINCT asgn_sub.userid
                FROM {assign_submission} asgn_sub
                JOIN {assign} a ON a.id = asgn_sub.assignment
                LEFT JOIN {assign_grades} ag ON ag.assignment = asgn_sub.assignment AND ag.userid = asgn_sub.userid
                    AND asgn_sub.attemptnumber = ag.attemptnumber
                WHERE a.id = :iteminstance
                    AND asgn_sub.status = 'submitted'
                    AND asgn_sub.userid > 0
                    AND a.grade <> 0 AND (ag.id IS NULL OR asgn_sub.timemodified >= ag.timemodified)
                ";

        return $DB->get_fieldset_sql($sql, ['iteminstance' => $this->gi->iteminstance]);
    }

    /**
     * Get the users with graded results
     */
    protected function get_graded_users() {
        global $DB;
        $sql = "SELECT DISTINCT g.userid
                FROM {grade_grades} g
                LEFT JOIN {grade_items} gi on g.itemid = gi.id
                WHERE gi.itemmodule = 'assign' AND gi.iteminstance = :iteminstance
                    AND g.finalgrade IS NOT NULL"; // MAy turn out to be needed, dunno.
        return $DB->get_fieldset_sql($sql, ['iteminstance' => $this->gi->iteminstance]);
    }

    /**
     * Count the number of students wit ungraded submissions in this activity
     * @param array $courseuserids
     * @return int Number of students with ungraded submissions
     */
    public function count_ungraded($courseuserids = []) {
        $ungraded = $this->get_ungraded_submissions();

        if (count($courseuserids) > 0) {
            $ungraded = array_intersect($ungraded, $courseuserids);
        }
        return count($ungraded);
    }

    /**
     * Count the number of students wit (all) graded submissions in this activity
     * @param array $courseuserids
     * @return int Number of students with graded submission
     */
    public function count_graded($courseuserids = []) {
        $ungraded = $this->get_ungraded_submissions();
        $graded = $this->get_graded_users();

        if (count($courseuserids) > 0) {
            $ungraded = array_intersect($ungraded, $courseuserids);
            $graded = array_intersect($graded, $courseuserids);
        }

        // Determine how many id's have a grade, but also an ungraded submission.
        $dual = array_intersect($ungraded, $graded);
        // Subtract those from the graded count.
        return count($graded) - count($dual);
    }

    /**
     * Check if specified user has ungraded submission in this activity
     * @param mixed $userid User id of user to check
     * @return bool If specified user has ungraded submissions in this activity
     */
    public function has_ungraded_submission($userid) {
        $ungraded = $this->get_ungraded_submissions();
        return in_array($userid, $ungraded);
    }

}
