<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Scan submissions in assign activities
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan\local\ungradedscanners;

defined('MOODLE_INTERNAL') || die();
require_once($CFG->dirroot.'/question/engine/states.php'); // For reading question state.

/**
 * Activity scanner for quiz activity
 */
class quiz_scanner extends scanner_base {
    /**
     * Retrieve the ungraded submissions in this activity
     * @return array
     */
    protected function get_ungraded_submissions() {
        // Count all users who have one or more questions that still need grading.
        global $DB;

        // First find all question attempts that need grading.
        $sql = "SELECT qza.id as submissionid, qza.userid as userid,
                    qas.questionattemptid as attempt_id, qas.sequencenumber as sequencenumber
                FROM {question_attempt_steps} qas
                JOIN {question_attempts} qna ON qas.questionattemptid = qna.id
                JOIN {quiz_attempts} qza ON qna.questionusageid = qza.uniqueid
                WHERE qas.state = 'needsgrading' AND qza.quiz = :iteminstance";

        $rs = $DB->get_recordset_sql($sql, ['iteminstance' => $this->gi->iteminstance]);
        $submissions = [];
        foreach ($rs as $r) {
            // Now, check if .
            $maxstatesql = "SELECT MAX(qas.sequencenumber)
                            FROM {question_attempt_steps} qas
                            WHERE qas.questionattemptid = {$r->attempt_id}";
            $max = $DB->get_field_sql($maxstatesql);
            if ($r->sequencenumber == $max) {
                $submissions[$r->userid] = true; // Set array index based on user id, to avoid checking if value is in array.
            }
        }
        $rs->close();
        return array_keys($submissions);
    }

    /**
     * Count the number of students wit ungraded submissions in this activity
     * @param array $courseuserids
     * @return int Number of students with ungraded submissions
     */
    public function count_ungraded($courseuserids = []) {
        $ungraded = $this->get_ungraded_submissions();
        if (count($courseuserids) > 0) {
            $ungraded = array_intersect($ungraded, $courseuserids);
        }
        return count($ungraded);
    }

    /**
     * Count the number of students wit (all) graded submissions in this activity
     * @param array $courseuserids
     * @return int Number of students with graded submission
     */
    public function count_graded($courseuserids = []) {
        // Count all users who submitted one or more finished tests.
        global $DB;
        $sql = "SELECT DISTINCT g.userid
                FROM {grade_grades} g
                LEFT JOIN {grade_items} gi on g.itemid = gi.id
                WHERE gi.itemmodule = 'quiz' AND gi.iteminstance = :iteminstance
                AND g.finalgrade IS NOT NULL"; // MAy turn out to be needed, dunno.

        $graded = $DB->get_fieldset_sql($sql, ['iteminstance' => $this->gi->iteminstance]);

        if (count($courseuserids) > 0) {
            $graded = array_intersect($graded, $courseuserids);
        }

        return count($graded);
    }

    /**
     * Check if specified user has ungraded submission in this activity
     * @param mixed $userid User id of user to check
     * @return bool If specified user has ungraded submissions in this activity
     */
    public function has_ungraded_submission($userid) {
        $ungraded = $this->get_ungraded_submissions();
        return in_array($userid, $ungraded);
    }

}
