<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Webservice class for retrieving student studyplans
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan;
defined('MOODLE_INTERNAL') || die();

require_once($CFG->libdir.'/externallib.php');

use local_treestudyplan\local\helpers\webservicehelper;
use local_treestudyplan\task\autocohortsync;
use core_external\external_api;
use core_external\external_function_parameters;
use core_external\external_description;
use core_external\external_value;
use core_external\external_multiple_structure;
use core_external\external_single_structure;
use core\context;
use core\context\system as context_system;
use core\context\course as context_course;
use core\context\coursecat as context_coursecat;

require_once($CFG->libdir.'/badgeslib.php');
/**
 * Webservice class for retrieving student studyplans
 */
class studentstudyplanservice extends external_api {
    /**
     * Capability required to view studyplans of other users
     * @var string
     */
    const CAP_VIEWOTHER = "local/treestudyplan:viewuserreports";
    /**
     * Capability required to be linked as coach to a studyplan
     * @var string
     */
    const CAP_COACH = "local/treestudyplan:coach";
    /************************
     *                      *
     * list_user_studyplans *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function list_user_studyplans
     */
    public static function list_user_studyplans_parameters(): external_function_parameters {
        return new external_function_parameters([
            "userid" => new external_value(PARAM_INT, 'id of student', VALUE_DEFAULT),
        ]);
    }

    /**
     * Return value description for webservice function list_user_studyplans
     */
    public static function list_user_studyplans_returns(): external_description {
        return new external_multiple_structure(
            studyplan::simple_structure()
        );
    }

    /**
     * Return overview of the studyplans for a specific user
     * @param int $userid ID of user to check specific info for
     * @return array
     */
    public static function list_user_studyplans($userid) {
        $list = [];
        $studyplans = studyplan::find_for_user($userid);
        foreach ($studyplans as $studyplan) {
            // Only include studyplans in the context the user has permissions for.
            if (webservicehelper::has_capabilities(self::CAP_VIEWOTHER, $studyplan->context(), false)) {
                $list[] = $studyplan->simple_model($userid);
            }
        }
        return $list;
    }

    /************************
     *                      *
     * get_user_studyplan   *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function get_user_studyplan
     */
    public static function get_user_studyplan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "userid" => new external_value(PARAM_INT, 'id of user'),
            "studyplanid" => new external_value(PARAM_INT, 'id of specific studyplan to provide'),
        ] );
    }

    /**
     * Return value description for webservice function get_user_studyplan
     */
    public static function get_user_studyplan_returns(): external_description {
        return studyplan::user_structure();
    }

    /**
     * Get a specific studyplan for a given user
     * @param int $userid ID of user to check specific info for
     * @param int $studyplanid ID of studyplan to view
     * @return array|null
     */
    public static function get_user_studyplan($userid, $studyplanid) {
        $studyplan = studyplan::find_by_id($studyplanid);
        if ($studyplan->is_coach() && !$studyplan->suspended()) {
            external_api::validate_context($studyplan->context());
        } else {
            webservicehelper::require_capabilities(self::CAP_VIEWOTHER, $studyplan->context());
        }

        if ($studyplan->exist_for_user($userid)) {
            preloader::preload_for_users($studyplan, $userid); // Preload most required data to avoid excessive database calls.
            $model = $studyplan->user_model($userid);
            return $model;
        } else {
            return null;
        }
    }

    /****************************
     *                          *
     * list_invited_studyplan    *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function list_invited_studyplan
     */
    public static function list_invited_studyplans_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "invitekey" => new external_value(PARAM_RAW, 'invite key'),
        ] );
    }

    /**
     * Return value description for webservice function list_invited_studyplan
     */
    public static function list_invited_studyplans_returns(): external_description {
        return new external_multiple_structure(
            studyplan::simple_structure()
        );
    }

    /**
     * List available studyplans based on invite key
     * @param int $invitekey Invitation key
     * @return array
     */
    public static function list_invited_studyplans($invitekey) {
        global $DB;
        // First check if studyplan sharing is enabled.
        if (!get_config("local_treestudyplan", "enableplansharing")) {
            return [];
        }

        $invite = $DB->get_record_select(
                        "local_treestudyplan_invit",
                        $DB->sql_compare_text("invitekey"). " = " . $DB->sql_compare_text(":invitekey"),
                        ['invitekey' => $invitekey]
                    );

        if (empty($invite)) {
            return [];
        }

        // Do not validate the context in this instance to avoid requiring logins when this is not wanted.

        $userid = $invite->user_id;

        $list = [];
        $studyplans = studyplan::find_for_user($userid);
        foreach ($studyplans as $studyplan) {
            $list[] = $studyplan->simple_model($userid);
        }
        return $list;
    }

    /****************************
     *                          *
     * get_invited_studyplan    *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function get_invited_studyplan
     */
    public static function get_invited_studyplan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "invitekey" => new external_value(PARAM_RAW, 'invite key'),
            "studyplanid" => new external_value(PARAM_INT, 'studyplan_id'),
        ] );
    }

    /**
     * Return value description for webservice function get_invited_studyplan
     */
    public static function get_invited_studyplan_returns(): external_description {
        return studyplan::user_structure();
    }

    /**
     * Get studyplan based on invite key and studyplan id
     * @param string $invitekey Invitation key
     * @param int $studyplanid ID of the studyplan to retrieve
     * @return array
     */
    public static function get_invited_studyplan($invitekey, $studyplanid) {
        global $DB;
        // First check if studyplan sharing is enabled.
        if (!get_config("local_treestudyplan", "enableplansharing")) {
            return [];
        }
        $invite = $DB->get_record_select(
                        "local_treestudyplan_invit",
                        $DB->sql_compare_text("invitekey"). " = " . $DB->sql_compare_text(":invitekey"),
                        ['invitekey' => $invitekey]
                    );

        if (empty($invite)) {
            return [];
        }

        // Do not validate the context in this instance to avoid requiring logins when this is not wanted.

        $studyplan = studyplan::find_by_id($studyplanid);
        $userid = $invite->user_id;

        if ($studyplan->exist_for_user($userid)) {
            if (!$studyplan->suspended()) {
                preloader::preload_for_users($studyplan, $userid); // Preload most required data to avoid excessive database calls.
                return $studyplan->user_model($userid);
            } else {
                throw new \moodle_exception("The selected studyplan is suspended");
            }

        } else {
            throw new \moodle_exception("Invitation's user is not linked to this studyplan");
        }
    }

    /************************
     *                      *
     * list_own_studyplans *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function list_own_studyplans
     */
    public static function list_own_studyplans_parameters(): external_function_parameters {
        return new external_function_parameters([]);
    }

    /**
     * Return value description for webservice function list_own_studyplans
     */
    public static function list_own_studyplans_returns(): external_description {
        return new external_multiple_structure(
            studyplan::simple_structure()
        );
    }

    /**
     * Return overview of the studyplans the current user is linked to
     * @return array
     */
    public static function list_own_studyplans() {
        global $USER;
        $userid = $USER->id;

        $list = [];
        $studyplans = studyplan::find_for_user($userid);
        foreach ($studyplans as $studyplan) {
            $list[] = $studyplan->simple_model($userid);
        }
        return $list;
    }

    /************************
     *                      *
     * get_own_studyplan   *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function get_own_studyplan
     */
    public static function get_own_studyplan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "studyplanid" => new external_value(PARAM_INT, 'id of studyplan to retrieve'),
        ] );
    }

    /**
     * Return value description for webservice function get_own_studyplan
     */
    public static function get_own_studyplan_returns(): external_description {
        return studyplan::user_structure();
    }

    /**
     * Get a studyplans for the current user
     * @param int $studyplanid Id of specific studyplan
     * @return array
     */
    public static function get_own_studyplan($studyplanid) {
        global $USER;

        // Validate this call in the system context.
        external_api::validate_context(context_system::instance());

        $userid = $USER->id;

        $studyplan = studyplan::find_by_id($studyplanid);
        if ($studyplan->exist_for_user($userid) && !$studyplan->suspended()) {
            preloader::preload_for_users($studyplan, $userid);
            $model = $studyplan->user_model($userid);
            return $model;
        } else {
            throw new \moodle_exception("You do not have access to this studyplan");
        }
    }

    /***************************
     *                         *
     * list_teaching_studyplans *
     *                         *
     ***************************/

    /**
     * Parameter description for webservice function list_teaching_studyplans
     */
    public static function list_teaching_studyplans_parameters(): external_function_parameters {
        return new external_function_parameters( [] );
    }

    /**
     * Return value description for webservice function list_teaching_studyplans
     */
    public static function list_teaching_studyplans_returns(): external_description {
        return new external_multiple_structure(
            studyplan::simple_structure()
        );
    }

    /**
     * Get all studyplans the current user is teaching in
     * @return array
     */
    public static function list_teaching_studyplans() {
        global $USER;
        $userid = $USER->id;

        external_api::validate_context(context_system::instance());
        $studyplans = teachingfinder::list_my_plans();

        $list = [];
        foreach ($studyplans as $studyplan) {
            $list[] = $studyplan->simple_model($userid);
        }
        return $list;
    }

    /***************************
     *                         *
     * list_coaching_studyplans *
     *                         *
     ***************************/

    /**
     * Parameter description for webservice function list_teaching_studyplans
     */
    public static function list_coaching_studyplans_parameters(): external_function_parameters {
        return new external_function_parameters( [] );
    }

    /**
     * Return value description for webservice function list_teaching_studyplans
     */
    public static function list_coaching_studyplans_returns(): external_description {
        return new external_multiple_structure(
            studyplan::simple_structure()
        );
    }

    /**
     * Get all studyplans the current user is coaching
     * @return array
     */
    public static function list_coaching_studyplans() {
        global $USER, $DB;
        $userid = $USER->id;

        external_api::validate_context(context_system::instance());

        // Return empty list if coach role not enabled.
        if (! (\get_config("local_treestudyplan", "enablecoach"))) {
            return [];
        }

        $records = $DB->get_records("local_treestudyplan_coach", ["user_id" => $userid]);

        $list = [];
        foreach ($records as $r) {
            try {
                $studyplan = studyplan::find_by_id($r->studyplan_id);
                if (has_capability(self::CAP_COACH, $studyplan->context(), $USER)) {
                    $list[] = $studyplan->simple_model_coach();
                }
            } catch (\Exception $x) {
                $studyplan = null;
            }
        }
        return $list;
    }

    /***************************
     *                         *
     * get_teaching_studyplan  *
     *                         *
     ***************************/

    /**
     * Parameter description for webservice function get_teaching_studyplan
     */
    public static function get_teaching_studyplan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "studyplanid" => new external_value(PARAM_INT, 'id of specific studyplan to provide', VALUE_DEFAULT),
        ] );
    }

    /**
     * Return value description for webservice function get_teaching_studyplan
     */
    public static function get_teaching_studyplan_returns(): external_description {
        return studyplan::teacher_structure();
    }

    /**
     * Get all or one studyplan the current user is teaching in
     * @param int $studyplanid ID of studyplan to retrieve
     * @return array
     */
    public static function get_teaching_studyplan($studyplanid) {
        external_api::validate_context(context_system::instance());
        $studyplan = studyplan::find_by_id($studyplanid);

        if (teachingfinder::am_teaching_studyplan($studyplan)) {
            preloader::preload_all($studyplan); // Preload most required data to avoid excessive database calls.
            return $studyplan->teacher_model();
        } else {
            throw new \moodle_exception("You are not teaching in this studyplan");
        }
    }

    /***************************
     *                         *
     * line_enrol_self         *
     *                         *
     ***************************/

    /**
     * Parameter description for webservice function list_enrol_self
     */
    public static function line_enrol_self_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of specific studyline to enrol into'),
        ] );
    }

    /**
     * Return value description for webservice function list_enrol_self
     */
    public static function line_enrol_self_returns(): external_description {
        return studyline::enrol_info_structure();
    }

    /**
     * Get all or one studyplan the current user is teaching in
     * @param int $id ID of studyplan to retrieve
     * @return array
     */
    public static function line_enrol_self($id) {
        global $USER;
        $userid = $USER->id;

        $o = studyline::find_by_id($id);
        if ($o->can_enrol($userid)) {
            $o->enrol($userid);
        }

        // Trigger immediate cohort synchronization for this line only.
        autocohortsync::syncline($o);

        return $o->enrol_info_model($userid);
    }

}
