/* eslint no-var: "error" */
/* eslint no-unused-vars: "off" */
/* eslint linebreak-style: "off" */
/* eslint no-trailing-spaces: "off" */
/* eslint-env es6*/

import {call} from 'core/ajax';
import notification from 'core/notification';

import Vue from './vue/vue';

import Debugger from './util/debugger';
import {loadStrings, decodeHtml} from './util/string-helper';

import SRComponents from './studyplan-report-components';
Vue.use(SRComponents);
import RVComponents from './report-viewer-components';
Vue.use(RVComponents);

import ModalComponents from './modedit-modal';
Vue.use(ModalComponents);

import PortalVue from './portal-vue/portal-vue.esm';
Vue.use(PortalVue);
import BootstrapVue from './bootstrap-vue/bootstrap-vue';
Vue.use(BootstrapVue);

let debug = new Debugger("treestudyplanviewer");

let strings = loadStrings({
    studyplanReport: {
        studyplan: 'studyplan',
        page: 'studyplanpage',
        periods: 'periods',
        period: 'period',
        loading: 'loading@core',
        all: 'all@core',
        from: 'from@core',
        to: 'to@core',
    },
});

/**
 * Initialize the Page
 * @param {string} selector The selector used to match html element to code
 * @param {Number} studyplanid The id of the studyplan we need to view 
 * @param {Number} pageid The id of the studyplan page we need to view 
 * @param {string} contextname The name of the current context
 * @param {string} studyplanname The name of the current studyplan
 * @param {string} pagename The name of the current page
 * @param {Number} firstperiod The number of the first period to view
 * @param {Number} lastperiod The number of the last period to view 
 */
export function init(selector, studyplanid, pageid, contextname, studyplanname, 
                     pagename, firstperiod, lastperiod) {
    if (undefined === pageid || !Number.isInteger(Number(pageid)) ||
        undefined === studyplanid || !Number.isInteger(Number(studyplanid))) {
        debug.error("Error: studyplan id and page id not provided as integer numbers to script.",
            studyplanid, pageid, firstperiod, lastperiod);
        return; // Do not continue if plan and page are not proper integers
    }
    // Ensure a numeric value instead of string.
    studyplanid = Number(studyplanid);
    pageid = Number(pageid);

    // Startup app.
    const app = new Vue({
        el: selector,
        data: {
            structure: null,
            studyplan: null,
            page: null,
            text: strings.studyplanReport,
            contextname: decodeHtml(contextname),
            studyplanname: decodeHtml(studyplanname),
            pagename: decodeHtml(pagename),
        },
        created() {
            // On creation, load the page as specified
            this.loadStructure(pageid, firstperiod, lastperiod);
        },
        methods: {
            loadStructure(pageid, firstperiod, lastperiod) {
                const self = this;
                this.structure = null; // Starts loading icon. Hides old data.
                call([{
                    methodname: 'local_treestudyplan_get_report_structure',
                    args: {pageid: pageid,
                        firstperiod: firstperiod,
                        lastperiod: lastperiod
                    }
                }])[0].then((response) => {
                    self.structure = response;
                    self.studyplan = response.studyplan;
                    self.page = response.page;
                    return;
                }).catch(notification.exception);
            },
            selectedPage(e) {
                debug.info("SelectedPage", e);
                const pageid = e.target.value;
                this.loadStructure(pageid);
            },
            selectedFirstPeriod(e) {
                debug.info("selectedFirstPeriod", e);
                let f = e.target.value;
                let l = this.structure.lastperiod;
                if (l < f) {
                    l = f;
                }
                this.loadStructure(this.page.id, f, l);
            },
            selectedLastPeriod(e) {
                debug.info("selectedLastPeriod", e);
                let f = this.structure.firstperiod;
                let l = e.target.value;
                if (l < f) {
                    l = f;
                }
                this.loadStructure(this.page.id, f, l);

            },
        },
        template: `
        <div class="local_treestudyplan_result_overview">
            <h3><b><span v-html="contextname"></span></b> / <span v-html="studyplanname"></span></h3>
            <h4 v-if="pagename != studyplanname"><span v-html="pagename"></span></h4>
            <div class='vue-loader' v-if='!structure'>
                <div class='spinner-border text-primary' role='status'>
                    <span class='sr-only'>Loading...</span>
                </div>
            </div>
            <template v-else>
                <div class='container q-pageperiodselection'>
                    <div class='row'>
                        <div class='col-sm-2 q-label'>{{text.page}}</div>
                        <div class='col-sm-6'>
                            <select class='custom-select' :value='page.id' @change='selectedPage'>
                                <template v-if='studyplan'>
                                    <option v-for='p in studyplan.pages'
                                        :key='p.id'
                                        :value='p.id'
                                        :selected='(page.id == p.id) ? true : false'
                                        ><span v-html="p.display.fullname"></span></option>
                                </template>
                                <option v-else disabled>{{text.loading}}</option>
                            </select>
                        </div>
                    </div>
                    <div class='row'>
                        <div class='col-sm-2 q-label' >{{text.period}} ({{text.from}} / {{text.to}})</div>
                        <div class='col-sm-2'>
                            <select class='custom-select' @change='selectedFirstPeriod'>
                                <template v-if='page'>
                                    <option v-for='p in page.perioddesc'
                                        :key='p.id'
                                        :value='p.period'
                                        :selected='(structure.firstperiod == p.period) ? true : false'
                                        ><span v-html="p.display.fullname"></span></option>
                                </template>
                                <option v-else disabled>{{text.loading}}</option>
                            </select>
                        </div>
                        <div class='col-sm-2 '>
                            <select class='custom-select' @change='selectedLastPeriod'>
                                <template v-if='page'>
                                    <template v-for='p in page.perioddesc'>
                                        <option v-if='p.period >= structure.firstperiod'
                                            :key='p.id'
                                            :value='p.period'
                                            :selected='(structure.lastperiod == p.period) ? true : false'
                                            ><span v-html="p.display.fullname"></span></option>
                                    </template>
                                </template>
                                <option v-else disabled>{{text.loading}}</option>
                            </select>
                        </div>
                    </div>
                </div>
                <q-studyplanreport :structure='structure' ></q-studyplanreport>
            </template>
        </div>`
    });
}
