<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Completion model
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan;

use core_external\external_description;
use core_external\external_value;

defined('MOODLE_INTERNAL') || die();

require_once($CFG->libdir.'/externallib.php');

/**
 * Model of different completion states
 */
class completion {
    /** @var int */
    public const FAILED = -1;
    /** @var int */
    public const INCOMPLETE = 0;
    /** @var int */
    public const PENDING = 1;
    /** @var int */
    public const PROGRESS = 2;
    /** @var int */
    public const COMPLETED = 3;
    /** @var int */
    public const GOOD = 4;
    /** @var int */
    public const EXCELLENT = 5;

    /**
     * Map of completion state constants to string labels
     * @var array
     */
    private const LABELS = [
        self::FAILED     => 'failed',
        self::INCOMPLETE => 'incomplete',
        self::PENDING    => 'pending',
        self::PROGRESS   => 'progress',
        self::COMPLETED  => 'completed',
        self::GOOD       => 'good',
        self::EXCELLENT  => 'excellent',
    ];

    /**
     * Convert completion state to equivalent text label
     * @param int $completion
     * @return string
     */
    public static function label($completion) {
        if (array_key_exists($completion, self::LABELS)) {
            return self::LABELS[$completion];
        } else {
            return self::LABELS[self::INCOMPLETE];
        }
    }

    /**
     * Webservice structure for basic info
     * @param int $value Webservice requirement constant
     */
    public static function structure($value = VALUE_REQUIRED): external_description {
        return new external_value( PARAM_TEXT,
                                    'completion state (failed|incomplete|pending|progress|completed|good|excellent)',
                                    $value);
    }

    /**
     * Count how many times each state occurs in a given array of completion states
     * @param int[] $states The array of completion states
     * @return array of state counts
     */
    public static function count_states(array $states) {
        // Initialize result array.
        $statecount = [];
        foreach (array_keys(self::LABELS) as $key) {
            $statecount[$key] = 0;
        }

        // Process all states in array and increment relevant counter for each one.
        foreach ($states as $c) {
            if (array_key_exists($c, $statecount)) {
                $statecount[$c] += 1;
            }
        }

        return $statecount;
    }

}
