<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Class to abstract information about (category) context for web service export
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan;
use core_external\external_api;
use core_external\external_function_parameters;
use core_external\external_description;
use core_external\external_value;
use core_external\external_multiple_structure;
use core_external\external_single_structure;
use core\context;
use core\context\system as context_system;
use local_treestudyplan\local\helpers\formatter;

/**
 * Class to abstract information about (category) context for web service export
 */
class contextinfo {
    /** @var context */
    public $context;
    /** @var array */
    public $ctxpath;

    /**
     * Create new based on context
     * @param context $context The context to describe
     */
    public function __construct(context $context) {
        $this->context = $context;
        $this->ctxpath = array_reverse($this->context->get_parent_context_ids(true));
        if (count($this->ctxpath) > 1 && $this->ctxpath[0] == 1) {
            array_shift($this->ctxpath);
        }
    }

    /**
     * Describe the result for the webservice model
     * @param int $value Webservice requirement constant
     */
    public static function structure($value = VALUE_REQUIRED): external_description {
        return new external_single_structure([
            "name" => new external_value(PARAM_RAW, 'context name'),
            "path" => new external_multiple_structure( new external_value(PARAM_RAW)),
        ], 'context information', $value);
    }

    /**
     * Make the webservice result model
     * @return array Webservice value
     */
    public function model() {

        return  [
            "name" => formatter::format_string($this->context->get_context_name(false, false)),
            "path" => $this->path(),
        ];
    }

    /**
     * Return context path names
     * @return array of context path names
     */
    public function path() {
        return array_map(function($c) {
            return formatter::format_string(((object)context::instance_by_id($c))->get_context_name(false, false));
        }, $this->ctxpath);
    }

    /**
     * Return full context path name
     * @return string Concatenated string of paths
     */
    public function pathstr() {
        return implode(" / ", $this->path());
    }

    /**
     * Make new Contextinfo for context id
     * @param int $contextid Context id
     */
    public static function by_id($contextid): self {
        return new self(self::context_by_id($contextid));
    }

    /**
     * Find context by id, but also return system context for context id 0
     * @param int $contextid Context id
     * @return context The context
     */
    public static function context_by_id($contextid): context {
        if ($contextid <= 1) {
            $contextid = 1;
        }

        $ctx = context::instance_by_id($contextid);
        if (is_object($ctx)) {
            return $ctx;
        } else {
            return context_system::instance();
        }
    }

}
