<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Handles the hooks for adding navigation items
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan\local;
use core\hook\navigation\primary_extend;
use moodle_url;
use navigation_node;
use core\context;
use core\context\system as context_system;
use core\context\coursecat as context_coursecat;
use local_treestudyplan\local\helpers\webservicehelper;
use local_treestudyplan\studyplan;

/**
 * Handles the hooks for adding navigation items
 */
class navigationhandler {
    /**
     * Hook callback function to extend the primary navigation.
     *
     * @param primary_extend $hook the primary_extend hook object
     */
    public static function extend_primary_navigation(primary_extend $hook): void {
        global $USER, $CFG, $DB;
        if (empty(get_config("local_treestudyplan", "primary_nav_items"))) {
            return;
        }
        // Retrieve navigation view from hook.
        $navigation = $hook->get_primaryview();
        $nodes = [];
        if ($USER->id > 1) {
            // Don't show if user is not logged in (id == 0) or is guest user (id == 1).
            $userstudyplans = studyplan::find_for_user($USER->id);
            if (!empty($userstudyplans)) {
                // Create studyplan node.
                $nodes[] = navigation_node::create(
                    get_string("link_myreport", "local_treestudyplan"),
                    new moodle_url($CFG->wwwroot . "/local/treestudyplan/myreport.php", []),
                );
            }

            if (has_capability('local/treestudyplan:viewuserreports', context_system::instance())
                || webservicehelper::has_capability_in_any_category('local/treestudyplan:viewuserreports')) {
                $nodes[] = navigation_node::create(
                    get_string("link_viewplan", "local_treestudyplan"),
                    new moodle_url($CFG->wwwroot . "/local/treestudyplan/view-plan.php", []),
                );
            }

            if (has_capability('local/treestudyplan:editstudyplan', context_system::instance())
                  || webservicehelper::has_capability_in_any_category('local/treestudyplan:editstudyplan')
               ) {
                $nodes[] = navigation_node::create(
                    get_string("link_editplan", "local_treestudyplan"),
                    new moodle_url($CFG->wwwroot . "/local/treestudyplan/edit-plan.php", []),
                );
            }

            $coachsql = "SELECT COUNT('id') FROM {local_treestudyplan_coach} c
                         INNER JOIN {local_treestudyplan} t ON c.studyplan_id = t.id
                         WHERE c.user_id = :user_id";

            if (\get_config("local_treestudyplan", "enablecoach") &&
                 (has_capability('local/treestudyplan:coach', context_system::instance())
                 || webservicehelper::has_capability_in_any_category('local/treestudyplan:coach')
                 ) && $DB->count_records_sql($coachsql, ["user_id" => $USER->id]) > 0
                ) {
                $nodes[] = navigation_node::create(
                    get_string("link_coach", "local_treestudyplan"),
                    new moodle_url($CFG->wwwroot . "/local/treestudyplan/coach.php", []),
                );
            }
        }

        if (\get_config("local_treestudyplan", "primary_nav_combine") && count($nodes) > 1) {
            // Create a node for the drop-down menu.
            $dropdown = navigation_node::create(
                get_string("link_studyplans", "local_treestudyplan"),
            );

            // Add all the nodes to the dropdown.
            foreach ($nodes as $node) {
                $dropdown->add_node($node);
            }

            // Finally, add the dropdown node.
            $navigation->add_node($dropdown);

        } else {
            // Just add all the nodes if they should not be combined.
            foreach ($nodes as $node) {
                $navigation->add_node($node);
            }
        }
    }
}
