<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Webservice class for managing studyplans
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan;
defined('MOODLE_INTERNAL') || die();

use local_treestudyplan\local\helpers\webservicehelper;
use local_treestudyplan\task\autocohortsync;
use core_external\external_api;
use core_external\external_function_parameters;
use core_external\external_description;
use core_external\external_value;
use core_external\external_multiple_structure;
use core_external\external_single_structure;
use core\context\system as context_system;
use core\context\course as context_course;
use core\context\coursecat as context_coursecat;

require_once($CFG->libdir.'/externallib.php');
require_once($CFG->libdir.'/badgeslib.php');
require_once($CFG->libdir.'/gradelib.php');
require_once($CFG->dirroot.'/course/modlib.php');

/**
 * Webservice class for managing studyplans
 */
class studyplanservice extends external_api {
    /**
     * Capability required to edit study plans
     * @var string
     */
    const CAP_EDIT = "local/treestudyplan:editstudyplan";
    /**
     * Capability required to view studyplans (for other users)
     * @var string
     */
    const CAP_VIEW = "local/treestudyplan:viewuserreports";
        /**
     * Capability required to view studyplans (for other users)
     * @var string
     */
    const CAP_COACH = "local/treestudyplan:coach";

    /************************
     *                      *
     * list_studyplans      *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function list_studyplans
     */
    public static function list_studyplans_parameters(): external_function_parameters {
        return new external_function_parameters([
            "context_id" => new external_value(PARAM_INT, 'context to search in for studyplans', VALUE_DEFAULT),
        ]);
    }

    /**
     * Return value description for webservice function list_studyplans
     */
    public static function list_studyplans_returns(): external_description {
        return new external_multiple_structure( studyplan::simple_structure() );
    }

    /**
     * Get overview of all studyplans in a given context
     * @param int $contextid Id of context to view (defaults to systemcontext)
     * @return array
     */
    public static function list_studyplans($contextid = 0) {
        // Check if the user has the correct permissions for this context.
        $context = webservicehelper::find_context($contextid);
        webservicehelper::require_capabilities([self::CAP_EDIT, self::CAP_VIEW], $context);

        // Now list all the studyplans in the relevant context.
        $list = [];
        $studyplans = studyplan::find_all($contextid);
        foreach ($studyplans as $studyplan) {
            $list[] = $studyplan->simple_model();
        }
        return $list;

    }

    /************************
     *                      *
     * get_studyplan_map    *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function get_studyplan_map
     */
    public static function get_studyplan_map_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of a studyplan to load', VALUE_REQUIRED),
        ] );
    }

    /**
     * Return value description for webservice function get_studyplan_map
     */
    public static function get_studyplan_map_returns(): external_description {
        return  studyplan::editor_structure();
    }

    /**
     * Get editor model for specific studyplan
     * @param int $id Id of studyplan
     * @return array|null
     */
    public static function get_studyplan_map($id) {
        if (isset($id) && $id > 0) {
            $studyplan = studyplan::find_by_id($id);
            preloader::preload_map($studyplan); // Preload most required data to avoid excessive database calls.

            if ($studyplan->is_coach() && !$studyplan->suspended()) {
                external_api::validate_context($studyplan->context());
            } else {
                webservicehelper::require_capabilities([self::CAP_EDIT], $studyplan->context());
            }

            // If suspended, only allow the mapping if the user has edit rights.
            if (!has_capability(self::CAP_EDIT, $studyplan->context()) && $studyplan->suspended()) {
                return null;
            } else {
                $model = $studyplan->editor_model();
                return $model;
            }
        } else {
            return null;
        }
    }

    /****************************
     *                          *
     * get_studyplan_teacehrmap *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function get_studyplan_resultmap
     */
    public static function get_studyplan_resultmap_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of a studyplan to load', VALUE_REQUIRED),
        ] );
    }

    /**
     * Return value description for webservice function get_studyplan_resultmap
     */
    public static function get_studyplan_resultmap_returns(): external_description {
        return studyplan::teacher_structure();
    }

    /**
     * Get teacher model for specific studyplan
     * @param int $id Id of studyplan
     * @return array|null
     */
    public static function get_studyplan_resultmap($id) {
        if (isset($id) && $id > 0) {
            $studyplan = studyplan::find_by_id($id);
            preloader::preload_all($studyplan); // Preload most required data to avoid excessive database calls.

            if ($studyplan->is_coach() && !$studyplan->suspended()) {
                external_api::validate_context($studyplan->context());
            } else {
                webservicehelper::require_capabilities([self::CAP_VIEW], $studyplan->context());
            }

            // If suspended, disallow.
            if ($studyplan->suspended()) {
                return null;
            } else {
                $model = $studyplan->teacher_model();
                return $model;
            }
        } else {
            return null;
        }
    }

    /************************
     *                      *
     * get_studyline_map    *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function get_studyline_map
     */
    public static function get_studyline_map_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of a studyline to check usage on', VALUE_DEFAULT),
        ]);
    }

    /**
     * Return value description for webservice function get_studyline_map
     */
    public static function get_studyline_map_returns(): external_description {
        return new external_multiple_structure( studyline::editor_structure() );
    }

    /**
     * Get editor model for specific study line
     * @param int $id ID of study line
     * @return array
     */
    public static function get_studyline_map($id) {

        $o = studyline::find_by_id($id);
        webservicehelper::require_capabilities([self::CAP_EDIT, self::CAP_VIEW], $o->context());
        return $o->editor_model();
    }

    /************************
     *                      *
     * add_studyplan        *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function add_studyplan
     */
    public static function add_studyplan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "name" => new external_value(PARAM_TEXT, 'name of studyplan'),
            "shortname" => new external_value(PARAM_TEXT, 'shortname of studyplan'),
            "idnumber" => new external_value(PARAM_TEXT, 'idnumber of studyplan'),
            "description" => new external_value(PARAM_TEXT, 'description of studyplan'),
            "periods" => new external_value(PARAM_INT, 'number of periods in studyplan'),
            "startdate" => new external_value(PARAM_TEXT, 'start date of studyplan'),
            "enddate" => new external_value(PARAM_TEXT, 'end date of studyplan'),
            "aggregation" => new external_value(PARAM_TEXT, 'selected aggregator', VALUE_DEFAULT),
            "aggregation_config" => new external_value(PARAM_TEXT, 'config string for aggregator', VALUE_DEFAULT),
            "context_id" => new external_value(PARAM_INT, 'context of the study plan', VALUE_DEFAULT),
        ] );
    }

    /**
     * Return value description for webservice function add_studyplan
     */
    public static function add_studyplan_returns(): external_description {
        return studyplan::simple_structure();
    }

    /**
     * Add a new studyplan in a given context
     * @param mixed $name
     * @param mixed $shortname
     * @param mixed $idnumber
     * @param mixed $description
     * @param mixed $periods
     * @param mixed $startdate
     * @param mixed $enddate
     * @param string $aggregation
     * @param string $aggregationconfig
     * @param int $contextid
     * @return array
     */
    public static function add_studyplan($name, $shortname, $idnumber, $description, $periods,
                                         $startdate, $enddate, $aggregation = "bistate", $aggregationconfig = '', $contextid = 0) {
        // Check if we have the proper rights for the requested context.
        $context = webservicehelper::find_context($contextid);
        webservicehelper::require_capabilities(self::CAP_EDIT, $context);

        $o = studyplan::add([
            'name' => $name,
            'shortname' => $shortname,
            'idnumber' => $idnumber,
            'description' => $description,
            'periods' => $periods,
            'startdate' => $startdate,
            'enddate' => empty($enddate) ? null : $enddate,
            'aggregation' => $aggregation,
            'aggregation_config' => $aggregationconfig,
            'context_id' => $contextid,
        ]);
        return $o->simple_model();
    }

    /************************
     *                      *
     * edit_studyplan       *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function edit_studyplan
     */
    public static function edit_studyplan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of studyplan'),
            "name" => new external_value(PARAM_TEXT, 'name of studyplan'),
            "shortname" => new external_value(PARAM_TEXT, 'shortname of studyplan'),
            "idnumber" => new external_value(PARAM_TEXT, 'idnumber of studyplan'),
            "description" => new external_value(PARAM_TEXT, 'description of studyplan'),
            "periods" => new external_value(PARAM_INT, 'number of periods in studyplan'),
            "startdate" => new external_value(PARAM_TEXT, 'start date of studyplan'),
            "enddate" => new external_value(PARAM_TEXT, 'end date of studyplan'),
            "aggregation" => new external_value(PARAM_TEXT, 'selected aggregator', VALUE_DEFAULT),
            "aggregation_config" => new external_value(PARAM_TEXT, 'config string for aggregator', VALUE_DEFAULT),
            "context_id" => new external_value(PARAM_INT, 'context of the study plan', VALUE_DEFAULT),
        ] );
    }

    /**
     * Return value description for webservice function edit_studyplan
     */
    public static function edit_studyplan_returns(): external_description {
        return studyplan::simple_structure();
    }

    /**
     * Edit studyplan parameters
     * @param mixed $id
     * @param mixed $name
     * @param mixed $shortname
     * @param mixed $idnumber
     * @param mixed $description
     * @param mixed $periods
     * @param mixed $startdate
     * @param mixed $enddate
     * @param string $aggregation
     * @param string $aggregationconfig
     * @param int $contextid
     * @return array
     */
    public static function edit_studyplan($id, $name, $shortname, $idnumber, $description, $periods, $startdate,
                                          $enddate, $aggregation = "bistate", $aggregationconfig = '', $contextid = 0) {
        // Validate access in the intended context.
        $context = webservicehelper::find_context($contextid);
        // Do not validate the context in this case, just check the permissions in the specified context.
        webservicehelper::require_capabilities(self::CAP_EDIT, $context, false);

        // Also check the permissions in the context of the studyplan, in case it is not the same.
        $o = studyplan::find_by_id($id);
        webservicehelper::require_capabilities(self::CAP_EDIT, $o->context());

        $o->edit([
            'name' => $name,
            'shortname' => $shortname,
            'idnumber' => $idnumber,
            'description' => $description,
            'periods' => $periods,
            'startdate' => $startdate,
            'enddate' => $enddate,
            'aggregation' => $aggregation,
            'aggregation_config' => $aggregationconfig,
            'context_id' => $contextid,
        ]);
        return $o->simple_model();
    }

    /************************
     *                      *
     * delete_studyplan     *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function delete_studyplan
     */
    public static function delete_studyplan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of studyplan'),
            "force" => new external_value(PARAM_BOOL, 'id of studyplan', VALUE_DEFAULT),
        ] );
    }

    /**
     * Return value description for webservice function delete_studyplan
     */
    public static function delete_studyplan_returns(): external_description {
        return success::structure();
    }

    /**
     * Delete a studyplan
     * @param mixed $id Id of the studyplan
     * @param bool $force Force deletion, even though studyplan is not empty
     * @return array Succes/fail model
     */
    public static function delete_studyplan($id, $force = false) {
        $o = studyplan::find_by_id($id);
        // Validate if the requesting user has the right to edit the plan in it's current context.
        webservicehelper::require_capabilities(self::CAP_EDIT, $o->context());
        return $o->delete(!!$force)->model();
    }

    /************************
     *                      *
     * add_studyline        *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function add_studyline
     */
    public static function add_studyline_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "page_id" => new external_value(PARAM_INT, 'id of studyplan to add line to'),
            "name" => new external_value(PARAM_TEXT, 'shortname of studyline'),
            "shortname" => new external_value(PARAM_TEXT, 'idnumber of studyline'),
            "color" => new external_value(PARAM_TEXT, 'description of studyline'),
            "sequence" => new external_value(PARAM_INT, 'sequence of studyline'),
            "enrollable" => new external_value(PARAM_INT, 'type of enrollable', VALUE_DEFAULT),
            "enrolroles" => new external_multiple_structure(new external_value(PARAM_INT), 'enrolling roles', VALUE_DEFAULT),
        ] );
    }

    /**
     * Return value description for webservice function add_studyline
     */
    public static function add_studyline_returns(): external_description {
        return studyline::editor_structure();
    }

    /**
     * Add a new study line
     * @param mixed $pageid Id of the page to put it in
     * @param mixed $name Name of the line
     * @param mixed $shortname Shortname of the line
     * @param mixed $color Color of the line
     * @param mixed $sequence Order for new line
     * @param int|null $enrollable New enrollable setting (From studyline::ENROLLABLE_...)
     * @param array|null $enrolroles New list of roles that can enrol a student
     * @return array
     */
    public static function add_studyline($pageid, $name, $shortname, $color, $sequence, $enrollable=null, $enrolroles=null) {
        // Validate if the requesting user has the right to edit the plan in it's current context.
        $page = studyplanpage::find_by_id($pageid);
        webservicehelper::require_capabilities(self::CAP_EDIT, $page->studyplan()->context());

        $add = [
            'page_id' => $pageid,
            'name' => $name,
            'shortname' => $shortname,
            'color' => $color,
            'sequence' => $sequence,
        ];

        if (isset($enrollable) &&
            $enrollable >= studyline::ENROLLABLE_NONE &&
            $enrollable <= studyline::ENROLLABLE_SELF_ROLE
            ) {
            $add['enrollable'] = $enrollable;
        }

        if (isset($enrolroles)) {
            $add['enrolrole'] = implode(", ", $enrolroles);
        }
        $o = studyline::add($add);
        return $o->editor_model();
    }

    /************************
     *                      *
     * edit_studyline       *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function edit_studyline
     */
    public static function edit_studyline_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of studyline'),
            "name" => new external_value(PARAM_TEXT, 'shortname of studyline'),
            "shortname" => new external_value(PARAM_TEXT, 'idnumber of studyline'),
            "color" => new external_value(PARAM_TEXT, 'description of studyline'),
            "enrollable" => new external_value(PARAM_INT, 'type of enrollable', VALUE_DEFAULT),
            "enrolroles" => new external_multiple_structure(new external_value(PARAM_INT), 'enrolling roles', VALUE_DEFAULT),
        ] );
    }

    /**
     * Return value description for webservice function edit_studyline
     */
    public static function edit_studyline_returns(): external_description {
        return studyline::editor_structure();
    }

    /**
     * Edit studyline parameters
     * @param mixed $id Id of the line
     * @param mixed $name New name of the line
     * @param mixed $shortname New shortname of the line
     * @param mixed $color New color of the line
     * @param int|null $enrollable New enrollable setting (From studyline::ENROLLABLE_...)
     * @param array|null $enrolroles New list of roles that can enrol a student
     * @return array
     */
    public static function edit_studyline($id, $name, $shortname, $color, $enrollable=null, $enrolroles=null) {
        $o = studyline::find_by_id($id);
        // Validate if the requesting user has the right to edit the plan in it's current context.
        webservicehelper::require_capabilities(self::CAP_EDIT, $o->context());

        $edit = [
            'name' => $name,
            'shortname' => $shortname,
            'color' => $color,
        ];

        if (isset($enrollable) &&
            $enrollable >= studyline::ENROLLABLE_NONE &&
            $enrollable <= studyline::ENROLLABLE_SELF_ROLE
            ) {
            $edit['enrollable'] = $enrollable;
        }

        if (isset($enrolroles)) {
            $edit['enrolrole'] = implode(", ", $enrolroles);
        }

        $o->edit($edit);

        return $o->editor_model();
    }

    /************************
     *                      *
     * delete_studyline     *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function delete_studyline
     */
    public static function delete_studyline_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of studyline'),
        ] );
    }

    /**
     * Return value description for webservice function delete_studyline
     */
    public static function delete_studyline_returns(): external_description {
        return success::structure();
    }

    /**
     * Delete a study line
     * @param mixed $id Id of the studyline
     * @return array Success/fail model
     *
     */
    public static function delete_studyline($id) {
        $o = studyline::find_by_id($id);
        // Validate if the requesting user has the right to edit the plan in it's current context.
        webservicehelper::require_capabilities(self::CAP_EDIT, $o->context());
        return $o->delete()->model();
    }

    /************************
     *                      *
     * reorder_studylines   *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function reorder_studylines
     */
    public static function reorder_studylines_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "sequence" => new external_multiple_structure(
                new external_single_structure([
                    "id" => new external_value(PARAM_INT, 'id of studyline '),
                    "sequence" => new external_value(PARAM_INT, 'order of studyline'),
                ]),
            ),
        ] );
    }

    /**
     * Return value description for webservice function reorder_studylines
     */
    public static function reorder_studylines_returns(): external_description {
        return success::structure();
    }

    /**
     * Reorder study lines
     * @param int[] $resequence New order of study lines by id
     * @return array
     */
    public static function reorder_studylines($resequence) {
        // Validate if the requesting user has the right to edit the lines in it's current context.
        foreach ($resequence as $sq) {
            $o = studyline::find_by_id(($sq['id']));
            webservicehelper::require_capabilities(self::CAP_EDIT, $o->context());
        }

        return studyline::reorder($resequence)->model();
    }

    /***********************************
     * STUDYITEM FUNCTIONS
     ***********************************/

    /************************
     *                      *
     * get_studyitem        *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function get_studyitem
     */
    public static function get_studyitem_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of a study item to retrieve'),
        ] );
    }

    /**
     * Return value description for webservice function get_studyitem
     */
    public static function get_studyitem_returns(): external_description {
        return studyitem::editor_structure();
    }

    /**
     * Get editor model for study item
     * @param mixed $id ID of study item
     * @return array
     */
    public static function get_studyitem($id) {
        $o = studyitem::find_by_id($id);
        webservicehelper::require_capabilities([self::CAP_EDIT, self::CAP_VIEW], $o->context());
        return $o->editor_model();

    }

    /************************
     *                      *
     * add_studyitem        *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function add_studyitem
     */
    public static function add_studyitem_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "line_id" => new external_value(PARAM_INT, 'id of related study line'),
            "type" => new external_value(PARAM_TEXT, 'type of study item'),
            "details" => new external_single_structure([
                "conditions" => new external_value(PARAM_TEXT, 'conditions for completion', VALUE_OPTIONAL),
                "competency_id" => new external_value(PARAM_INT, 'id of referenced competency', VALUE_OPTIONAL),
                "course_id" => new external_value(PARAM_INT, 'id of referenced course', VALUE_OPTIONAL),
                "badge_id" => new external_value(PARAM_INT, 'id of referenced badge', VALUE_OPTIONAL),
                "continuation_id" => new external_value(PARAM_INT, 'id of continued item', VALUE_OPTIONAL),
            ]),
            "slot" => new external_value(PARAM_INT, 'slot in the study plan', VALUE_DEFAULT),
            "layer" => new external_value(PARAM_INT, 'layer in the slot', VALUE_OPTIONAL),
        ] );
    }

    /**
     * Return value description for webservice function add_studyitem
     */
    public static function add_studyitem_returns(): external_description {
        return studyitem::editor_structure();
    }

    /**
     * Add a new study item
     * @param int $lineid
     * @param string $type
     * @param array $details
     * @param int $slot
     * @param int $layer
     * @return array
     */
    public static function add_studyitem($lineid, $type, $details, $slot = -1, $layer = 0) {
        $line = studyline::find_by_id($lineid);
        $studyplan = $line->studyplan();
        if ($studyplan->is_coach() && !$studyplan->suspended()) {
            external_api::validate_context($studyplan->context());
        } else {
            webservicehelper::require_capabilities(self::CAP_EDIT, $studyplan->context());
        }

        $o = studyitem::add([
            'line_id' => $lineid,
            'type' => $type,
            'slot' => $slot,
            'layer' => $layer,
            'competency_id' => isset($details['competency_id']) ? $details['competency_id'] : null,
            'course_id' => isset($details['course_id']) ? $details['course_id'] : null,
            'badge_id' => isset($details['badge_id']) ? $details['badge_id'] : null,
            'continuation_id' => isset($details['continuation_id']) ? $details['continuation_id'] : null,
        ]);
        return $o->editor_model();
    }

    /************************
     *                      *
     * edit_studyitem       *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function edit_studyitem
     */
    public static function edit_studyitem_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of study item'),
            "conditions" => new external_value(PARAM_TEXT, 'conditions for completion'),
            "continuation_id" => new external_value(PARAM_INT, 'id of continued item', VALUE_DEFAULT),
        ]);
    }

    /**
     * Return value description for webservice function edit_studyitem
     */
    public static function edit_studyitem_returns(): external_description {
        return studyitem::editor_structure();
    }

    /**
     * Edit study item paremeters
     * @param mixed $id Id of studt item
     * @param mixed $conditions Conditions related to item (filters only)
     * @param bool $continuationid Link to previous filter item to copy result from (not used)
     * @return array
     */
    public static function edit_studyitem($id, $conditions, $continuationid = false) {

        $o = studyitem::find_by_id($id);
        $studyplan = $o->studyline()->studyplan();
        if ($studyplan->is_coach() && !$studyplan->suspended()) {
            external_api::validate_context($studyplan->context());
        } else {
            webservicehelper::require_capabilities(self::CAP_EDIT, $o->context());
        }

        $config = [
            'conditions' => $conditions,
        ];
        if ($continuationid !== false) {
            $config['continuation_id'] = $continuationid;
        }

        $o->edit($config);
        return $o->editor_model();

    }

    /************************
     *                      *
     * reorder_studyitems   *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function reorder_studyitems
     */
    public static function reorder_studyitems_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "items" => new external_multiple_structure(
                new external_single_structure([
                    "id" => new external_value(PARAM_INT, 'id of studyitem '),
                    "line_id" => new external_value(PARAM_INT, 'id of related study line'),
                    "slot" => new external_value(PARAM_INT, 'slot in the study plan'),
                    "layer" => new external_value(PARAM_INT, 'layer in the slot'),
                ]),
            ),
        ] );
    }

    /**
     * Return value description for webservice function reorder_studyitems
     */
    public static function reorder_studyitems_returns(): external_description {
        return success::structure();
    }

    /**
     * Reposition study items in line, layer and/or slot
     * @param mixed $resequence Array of item info [id, line_id, slot, layer]
     * @return array Success/fail model
     */
    public static function reorder_studyitems($resequence) {
        // Check for permissions to modify the studyplan.
        foreach ($resequence as $sq) {
            $item = studyitem::find_by_id(($sq['id']));
            $studyplan = $item->studyline()->studyplan();

            if ($studyplan->is_coach() && !$studyplan->suspended()) {
                external_api::validate_context($studyplan->context());
            } else {
                webservicehelper::require_capabilities(self::CAP_EDIT, $studyplan->context());
            }
        }

        return studyitem::reorder($resequence)->model();
    }

    /************************
     *                      *
     * delete_studyitem     *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function delete_studyitem
     */
    public static function delete_studyitem_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of studyitem'),
        ] );
    }

    /**
     * Return value description for webservice function delete_studyitem
     */
    public static function delete_studyitem_returns(): external_description {
        return success::structure();
    }

    /**
     * Delete a studyitem
     * @param mixed $id Id of study item to delete
     * @return array Success/fail model
     */
    public static function delete_studyitem($id) {
        $o = studyitem::find_by_id($id);
        $studyplan = $o->studyline()->studyplan();
        if ($studyplan->is_coach() && !$studyplan->suspended()) {
            external_api::validate_context($studyplan->context());
        } else {
            webservicehelper::require_capabilities(self::CAP_EDIT, $o->context());
        }

        return $o->delete()->model();
    }

    /************************
     *                      *
     * connect_studyitems   *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function connect_studyitems
     */
    public static function connect_studyitems_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "from_id" => new external_value(PARAM_INT, 'id of studyitem connect start '),
            "to_id" => new external_value(PARAM_INT, 'id ofstudyitem connect end'),
        ] );
    }

    /**
     * Return value description for webservice function connect_studyitems
     */
    public static function connect_studyitems_returns(): external_description {
        return studyitemconnection::structure();
    }

    /**
     * Connect two studylines
     * @param mixed $fromid Originating item
     * @param mixed $toid Target item
     * @return array Success/fail model
     */
    public static function connect_studyitems($fromid, $toid) {
        // Validate permissions.
        $studyplan = studyitem::find_by_id($fromid)->studyline()->studyplan();
        $toplan = studyitem::find_by_id($toid)->studyline()->studyplan();
        if ($toplan->id() != $studyplan->id()) {
            throw new \webservice_access_exception("The items to connect need to be in the same studyplan" );
        }
        if ($studyplan->is_coach() && !$studyplan->suspended()) {
            external_api::validate_context($studyplan->context());
        } else {
            webservicehelper::require_capabilities(self::CAP_EDIT, $studyplan->context());
        }

        $o = studyitemconnection::connect($fromid, $toid);
        return $o->model();
    }

    /****************************
     *                          *
     * disconnect_studyitems    *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function disconnect_studyitems
     */
    public static function disconnect_studyitems_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "from_id" => new external_value(PARAM_INT, 'id of studyitem '),
            "to_id" => new external_value(PARAM_INT, 'id of related study line'),
        ] );
    }

    /**
     * Return value description for webservice function disconnect_studyitems
     */
    public static function disconnect_studyitems_returns(): external_description {
        return success::structure();
    }

    /**
     * Disconnect two studylines
     * @param mixed $fromid Originating item
     * @param mixed $toid Target item
     * @return array Success/fail model
     */
    public static function disconnect_studyitems($fromid, $toid) {
        // Validate permissions.
        $studyplan = studyitem::find_by_id($fromid)->studyline()->studyplan();
        $toplan = studyitem::find_by_id($toid)->studyline()->studyplan();
        if ($toplan->id() != $studyplan->id()) {
            throw new \webservice_access_exception("The items to connect need to be in the same studyplan" );
        }
        if ($studyplan->is_coach() && !$studyplan->suspended()) {
            external_api::validate_context($studyplan->context());
        } else {
            webservicehelper::require_capabilities(self::CAP_EDIT, $studyplan->context());
        }

        return studyitemconnection::disconnect($fromid, $toid)->model();
    }

    /****************************
     *                          *
     * list badges              *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function list_badges
     */
    public static function list_badges_parameters(): external_function_parameters {
        return new external_function_parameters( [
        ] );
    }

    /**
     * Return value description for webservice function list_badges
     */
    public static function list_badges_returns(): external_description {
        return new external_multiple_structure(badgeinfo::editor_structure());
    }

    /**
     * List all available badges to drag into a studyplan page
     * @return array
     */
    public static function list_badges() {
        webservicehelper::system_context(); // Validate system context.

        $result = [];
        $badges = badges_get_badges(BADGE_TYPE_SITE, "timemodified");

        foreach ($badges as $badge) {
            $result[] = (new badgeinfo($badge))->editor_model();
        }
        return $result;

    }

    /****************************
     *                          *
     * search badges            *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function search_badges
     */
    public static function search_badges_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "search" => new external_value(PARAM_TEXT, 'search string', VALUE_DEFAULT),
        ] );
    }

    /**
     * Return value description for webservice function search_badges
     */
    public static function search_badges_returns(): external_description {
        return new external_multiple_structure(badgeinfo::simple_structure());
    }

    /**
     * List all available site badges to drag into a studyplan page
     * @param string $search Search string to use
     * @return array
     */
    public static function search_badges($search) {
        $systemcontext = webservicehelper::system_context();
        // Check system permission to.
        webservicehelper::require_capabilities("moodle/badges:viewbadges", $systemcontext);

        $results = badgeinfo::search_site_badges($search);
        return $results;
    }

    /**
     * Parameter description for webservice function search_related_badges
     */
    public static function search_related_badges_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "page_id" => new external_value(PARAM_INT, 'id of studyplan page'),
            "search" => new external_value(PARAM_TEXT, 'search string', VALUE_DEFAULT),
        ] );
    }

    /**
     * Return value description for webservice function search_related_badges
     */
    public static function search_related_badges_returns(): external_description {
        return new external_multiple_structure(badgeinfo::simple_structure());
    }

    /**
     * List all available badges to drag into a studyplan page
     * @param int $pageid ID of the studyplan page to limit search to
     * @param string $search Search string to use
     * @return array
     */
    public static function search_related_badges($pageid, $search="") {
        $results = [];
        $page = studyplanpage::find_by_id(($pageid));
        $studyplan = $page->studyplan();
        if ($studyplan->is_coach() && !$studyplan->suspended()) {
            external_api::validate_context($studyplan->context());
        } else {
            webservicehelper::require_capabilities(self::CAP_EDIT, $studyplan->context());
        }
        $results = badgeinfo::find_page_related_badges($page, $search);
        return $results;
    }

    /****************************
     *                          *
     * include/remove grades    *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function include_grade
     */
    public static function include_grade_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "grade_id" => new external_value(PARAM_INT, 'id of gradeitem '),
            "item_id" => new external_value(PARAM_INT, 'id of studyitem '),
            "include" => new external_value(PARAM_BOOL, 'include or not '),
            "required" => new external_value(PARAM_BOOL, 'required grade or not', VALUE_DEFAULT),
            ] );
    }

    /**
     * Return value description for webservice function include_grade
     */
    public static function include_grade_returns(): external_description {
        return success::structure();
    }

    /**
     * Mark a gradable item for inclusion in the studyplan
     * @param mixed $gradeid Id of gradable
     * @param mixed $itemid Id of study item
     * @param bool $include Include grade or not
     * @param bool $required Mark grade as required or not
     * @return array Success/Fail model
     */
    public static function include_grade($gradeid, $itemid, $include, $required = false) {
        global $USER;

        // Find related course and course context.
        $coursecontext = gradeinfo::get_coursecontext_by_id($gradeid);
        // Do sanity checks.

        $studyplan = studyitem::find_by_id($itemid)->studyline()->studyplan();
        external_api::validate_context($studyplan->context());
        // Check correct capabilities.
        if (($studyplan->is_coach() && !$studyplan->suspended())
            || has_capability(self::CAP_EDIT, $studyplan->context())
            || is_enrolled($coursecontext, $USER, 'local/treestudyplan:selectowngradables')) {
            return gradeinfo::include_grade($gradeid, $itemid, $include, $required)->model();
        } else {
            return success::fail("Access denied")->model();
        }

    }

    /****************************************
     *                                      *
     * mark/unmark competency required      *
     *                                      *
     ****************************************/

    /**
     * Parameter description for webservice function require_competency
     */
    public static function require_competency_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "competency_id" => new external_value(PARAM_INT, 'id of competency '),
            "item_id" => new external_value(PARAM_INT, 'id of studyitem '),
            "required" => new external_value(PARAM_BOOL, 'required grade or not', VALUE_DEFAULT),
            ] );
    }

    /**
     * Return value description for webservice function require_competency
     */
    public static function require_competency_returns(): external_description {
        return success::structure();
    }

    /**
     * Mark a competency as required for course completion
     * @param mixed $competencyid Id of gradable
     * @param mixed $itemid Id of study item
     * @param bool $required Mark grade as required or not
     * @return array Success/Fail model
     *
     */
    public static function require_competency($competencyid, $itemid, $required) {
        global $USER;
        $item = studyitem::find_by_id($itemid);
        external_api::validate_context($item->context());
        // Find related course and course context.
        if ($item->courseid()) {
            $coursecontext = context_course::instance($item->courseid());
        } else {
            $coursecontext = null;
        }

        // Check correct capabilities.
        $studyplan = $item->studyline()->studyplan();
        if (($studyplan->is_coach() && !$studyplan->suspended())
            || has_capability('local/treestudyplan:editstudyplan', $studyplan->context())
            || ($coursecontext && is_enrolled($coursecontext, $USER, 'local/treestudyplan:selectowngradables'))
        ) {
            return coursecompetencyinfo::require_competency($competencyid, $itemid, $required)->model();
        } else {
            return success::fail("Access denied")->model();
        }

    }

    /****************************
     *                          *
     * list aggregators         *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function list_aggregators
     */
    public static function list_aggregators_parameters(): external_function_parameters {
        return new external_function_parameters([]);
    }

    /**
     * Return value description for webservice function list_aggregators
     */
    public static function list_aggregators_returns(): external_description {
        return aggregator::list_structure();
    }

    /**
     * List available aggregators
     * @return array
     */
    public static function list_aggregators() {
        return aggregator::list_model();

    }

    /****************************
     *                          *
     * force_studyplan_scale    *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function force_studyplan_scale
     */
    public static function force_studyplan_scale_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "studyplan_id" => new external_value(PARAM_INT, 'id of studyplan'),
            "scale_id" => new external_value(PARAM_INT, 'scale_id to set'),
        ] );
    }

    /**
     * Return value description for webservice function force_studyplan_scale
     */
    public static function force_studyplan_scale_returns(): external_description {
        return new external_multiple_structure(new external_single_structure([
            "course" => courseinfo::simple_structure(),
            "grades" => new external_multiple_structure(new external_single_structure([
                "name" => new external_value(PARAM_TEXT, 'grade name'),
                "changed" => new external_value(PARAM_TEXT, 'changed or not'),
                "debug" => new external_value(PARAM_TEXT, 'debug', VALUE_OPTIONAL),
            ])),
        ]));
    }

    /**
     * Force all gradables in the studyplan to the same scale item
     * @param int $studyplanid Id of studyplan
     * @param int $scaleid Id of scale to use
     * @return array
     */
    public static function force_studyplan_scale($studyplanid, $scaleid) {
        global $DB;
        $dbman = $DB->get_manager();

        // Validate permissions.
        $context = studyplan::find_by_id($studyplanid)->context();
        webservicehelper::require_capabilities([self::CAP_EDIT, "local/treestudyplan:forcescales"], $context);

        $list = [];
        // Check if scaleid is valid.
        $scale = \grade_scale::fetch(['id' => $scaleid]);
        $scale->load_items();

        if ($scale) {
            $gradecfg = $DB->get_record("local_treestudyplan_gradecfg", ["scale_id" => $scale->id]);
            $scalepass = ($gradecfg) ? $gradecfg->min_completed : 0;
            $scalemax = count($scale->scale_items);

            // Find studyline id's.
            $studylineids = $DB->get_fieldset_select(studyline::TABLE, "id",
                                                     "studyplan_id = :plan_id", ['plan_id' => $studyplanid]);
            foreach ($studylineids as $studylineid) {
                // Find id's of studyitems of type course.
                $records = $DB->get_records(studyitem::TABLE, ['line_id' => $studylineid]);

                foreach ($records as $itemr) {
                    $studyitem = new studyitem($itemr->id, null);
                    if ($studyitem->type() == studyitem::COURSE) {
                        $courseinfo = $studyitem->getcourseinfo();
                        if (is_object($courseinfo)) {
                            $gradables = gradeinfo::list_studyitem_gradables($studyitem);

                            $gradelist = [];
                            foreach ($gradables as $g) {
                                $gi = $g->get_gradeitem();

                                // Only change items that do not yet have grades.
                                // Otherwise we will need to implement grade recalculations and it is not worth the trouble. .
                                // If grades are given, you likely don't want to change it like this anyway.

                                if (!$gi->has_grades()) {
                                    $gi->gradetype = GRADE_TYPE_SCALE;
                                    $gi->scaleid = $scale->id;
                                    $gi->grademin = 1;
                                    $gi->grademax = $scalemax;
                                    $gi->gradepass = $scalepass;

                                    // Update, signalling with our signature and bulkupdate.
                                    $result = $gi->update("local/treestudyplan");

                                    $debug = "";
                                    if ($result) {
                                        $updated = "converted";
                                    } else {
                                        $updated = "error";
                                    }

                                    // Next update the activity's table if it has a grade field.
                                    // Grade is generally set to the negative scale id if it is a scale.
                                    $tablename = $gi->itemmodule;
                                    $fieldname = "grade";
                                    if ($result && $gi->itemtype == "mod" && $dbman->table_exists($tablename)) {
                                        if ($dbman->field_exists($tablename, $fieldname)) {
                                            $gradevalue = intval(0 - ($scale->id));
                                            try {
                                                $DB->set_field($tablename, $fieldname, $gradevalue, ["id" => $gi->iteminstance]);
                                            } catch (\dml_exception $x) {
                                                $updated = "fail";
                                                $debug = strval($x);
                                            }
                                        }
                                    }

                                } else {
                                    $updated = "skipped";
                                }

                                $gradelist[] = [
                                    'name' => $gi->itemname,
                                    'changed' => $updated,
                                    'debug' => $debug,
                                ];
                            }

                            $list[] = [
                                'course' => $courseinfo->simple_model(),
                                'grades' => $gradelist,
                            ];
                        }
                    }
                }
            }
        }

        return $list;

    }

    /****************************
     *                          *
     * list_scales              *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function list_scales
     */
    public static function list_scales_parameters(): external_function_parameters {
        return new external_function_parameters( [] );
    }

    /**
     * Return value description for webservice function list_scales
     */
    public static function list_scales_returns(): external_description {
        return new external_multiple_structure(new external_single_structure([
            "id" => new external_value(PARAM_INT, 'id of scale'),
            "name" => new external_value(PARAM_TEXT, 'scale name'),
        ]));
    }

    /**
     * List available scales
     * @return array
     */
    public static function list_scales() {
        $list = [];
        $scales = \grade_scale::fetch_all_global();

        foreach ($scales as $scale) {
            $list[] = [
                "id" => $scale->id,
                "name" => $scale->name,
            ];
        }

        return $list;

    }

    /****************************
     *                          *
     * duplicate studyplan      *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function duplicate_plan
     */
    public static function duplicate_plan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "plan_id" => new external_value(PARAM_INT, 'id of plan to copy '),
            "name" => new external_value(PARAM_TEXT, 'name of copy '),
            "shortname" => new external_value(PARAM_TEXT, 'shortname of copy '),
            ] );
    }

    /**
     * Return value description for webservice function duplicate_plan
     */
    public static function duplicate_plan_returns(): external_description {
        return studyplan::simple_structure();
    }

    /**
     * Duplicate a studyplan into a new one
     * @param mixed $studyplanid Id of the plan to duplicate
     * @param mixed $name New fullname
     * @param mixed $shortname New shortname
     * @return array
     */
    public static function duplicate_plan($studyplanid, $name, $shortname) {
        // Validate permissions.
        webservicehelper::require_capabilities(self::CAP_EDIT, studyplan::find_by_id($studyplanid)->context());

        return studyplan::duplicate_plan($studyplanid, $name, $shortname);
    }

    /****************************
     *                          *
     * export studyplan         *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function export_plan
     */
    public static function export_plan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "studyplan_id" => new external_value(PARAM_INT, 'id of plan to export '),
            "format" => new external_value(PARAM_TEXT, 'export format', VALUE_OPTIONAL),
            ] );
    }

    /**
     * Return value description for webservice function export_plan
     */
    public static function export_plan_returns(): external_description {
        return studyplan::export_structure();
    }

    /**
     * Export studyplan
     * @param mixed $studyplanid Id of studyplan to export
     * @return array
     */
    public static function export_plan($studyplanid) {
        try {
            // Validate permissions.
            webservicehelper::require_capabilities(self::CAP_EDIT, studyplan::find_by_id($studyplanid)->context());

            $plan = studyplan::find_by_id($studyplanid);
            return $plan->export_plan();
        } catch (\webservice_access_exception $x) {
            return [ "format" => "", "content" => ""];
        }
    }

    /**
     * Parameter description for webservice function export_page
     */
    public static function export_page_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "page_id" => new external_value(PARAM_INT, 'id of plan to export '),
            "format" => new external_value(PARAM_TEXT, 'export format', VALUE_OPTIONAL),
            ] );
    }

    /**
     * Return value description for webservice function export_page
     */
    public static function export_page_returns(): external_description {
        return studyplanpage::export_structure();
    }

    /**
     * Export studylines for a studyplan page
     * @param mixed $pageid Id of the studyplan page to export the studylines for
     * @param string $format Export format [csv, json (default)]
     * @return array
     */
    public static function export_page($pageid, $format="json") {
        try {
            $page = studyplanpage::find_by_id($pageid);
            webservicehelper::require_capabilities(self::CAP_EDIT, $page->studyplan()->context());
            if ($format == "csv") {
                return $page->export_page_csv();
            } else {
                return $page->export_page();
            }
        } catch (\webservice_access_exception $x) {
            return [ "format" => "", "content" => ""];
        }
    }

    /****************************
     *                          *
     * import studyplan         *
     *                          *
     ****************************/

    /**
     * Parameter description for webservice function import_plan
     */
    public static function import_plan_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "content" => new external_value(PARAM_RAW, 'import file content'),
            "format" => new external_value(PARAM_TEXT, 'import format'),
            "context_id" => new external_value(PARAM_INT, 'context of the study plan', VALUE_DEFAULT),
            ] );
    }

    /**
     * Return value description for webservice function import_plan
     */
    public static function import_plan_returns(): external_description {
        return success::structure();
    }

    /**
     * Import studyplan from file
     * @param string $content Content of file
     * @param string $format Format of file
     * @param int $contextid ID of context to import to
     * @return array Success/fail model
     */
    public static function import_plan($content, $format = "application/json", $contextid = 1) {

        try {
            // Validate import context.
            $context = webservicehelper::find_context($contextid);
            webservicehelper::require_capabilities(self::CAP_EDIT, $context);

            $result = studyplan::import_studyplan($content, $format, $contextid);
            return (new success($result, "During study plan import"))->model();
        } catch (\webservice_access_exception $x) {
            return success::fail("Access denied")->model();
        }
    }

    /**
     * Parameter description for webservice function import_studylines
     */
    public static function import_pages_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "studyplan_id" => new external_value(PARAM_INT, 'id of studyplan to import to '),
            "content" => new external_value(PARAM_RAW, 'import file content'),
            "format" => new external_value(PARAM_TEXT, 'import format'),
            ] );
    }

    /**
     * Return value description for webservice function import_studylines
     */
    public static function import_pages_returns(): external_description {
        return success::structure();
    }

    /**
     * Import studylines into existing studtplan
     * @param int $studyplanid ID of studyplan to import to
     * @param string $content Content of file
     * @param string $format Format of file
     * @return array Success/fail model
     */
    public static function import_pages($studyplanid, $content, $format = "application/json") {
        try {
            $plan = studyplan::find_by_id($studyplanid);
            // Validate import context.
            webservicehelper::require_capabilities(self::CAP_EDIT, $plan->context());
            $result = $plan->import_pages($content, $format);
            return ($result ? success::success() : success::fail())->model();
        } catch (\webservice_access_exception $x) {
            return success::fail("Access denied")->model();
        }
    }

    /**
     * Parameter description for webservice function import_studylines
     */
    public static function import_studylines_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "page_id" => new external_value(PARAM_INT, 'id of studyplan page to import to '),
            "content" => new external_value(PARAM_RAW, 'import file content'),
            "format" => new external_value(PARAM_TEXT, 'import format'),
            ] );
    }

    /**
     * Return value description for webservice function import_studylines
     */
    public static function import_studylines_returns(): external_description {
        return success::structure();
    }

    /**
     * Import studylines into existing studtplan page
     * @param int $pageid ID of studyplan page to import to
     * @param string $content Content of file
     * @param string $format Format of file
     * @return array Success/fail model
     */
    public static function import_studylines($pageid, $content, $format = "application/json") {
        try {
            $page = studyplanpage::find_by_id($pageid);
            // Validate import context.
            webservicehelper::require_capabilities(self::CAP_EDIT, $page->studyplan()->context());

            $result = $page->import_studylines($content, $format);
            return ($result ? success::success() : success::fail())->model();
        } catch (\webservice_access_exception $x) {
            return success::fail("Access denied")->model();
        }
    }

    /********************************************************
     *                                                      *
     * Read and write course module title and desc          *
     * Used only in bistate aggregation method              *
     *                                                      *
     ********************************************************/

    /**
     * Parameter description for webservice function submit_cm_editform
     */
    public static function submit_cm_editform_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "cmid" => new external_value(PARAM_INT, 'id of course module'),
            "formdata" => new external_value(PARAM_RAW, 'url encoded form data'),
        ] );
    }

    /**
     * Return value description for webservice function submit_cm_editform
     */
    public static function submit_cm_editform_returns(): external_description {
        return success::structure();
    }

    /**
     * Submit hacked course activity edit form to edit just name and description
     * @deprecated will remove hacked edit form in the future
     * @param mixed $cmid
     * @param mixed $formdata
     * @return array Success/fail structure
     */
    public static function submit_cm_editform($cmid, $formdata) {
        global $CFG;

        // Check the course module exists.
        $cm = \get_coursemodule_from_id('', $cmid, 0, false, MUST_EXIST);
        // Get some context ready.
        $context = context_course::instance($cm->course);
        self::validate_context($context);

        // Check the course exists.
        $course = \get_course($cm->course);
        // Require_login.
        require_login($course, false, $cm); // Needed to setup proper $COURSE.

        // Get needed info to create the correct form.
        list($cm, $context, $module, $data, $cw) = \get_moduleinfo_data($cm, $course);
        $modmoodleform = "$CFG->dirroot/mod/$module->name/mod_form.php";
        if (file_exists($modmoodleform)) {
            require_once($modmoodleform);
        } else {
            throw new \moodle_exception('noformdesc', 'error');
        }
        $mformclassname = 'mod_'.$module->name.'_mod_form';

        // Now hack the received data into $_POST, so the mform thinks it has been submitted "normally".
        foreach (explode("&", $formdata) as $pair) {
            $p = explode("=", $pair, 2);
            $k = urldecode($p[0]);
            $v = (count($p) > 1) ? urldecode($p[1]) : "";

            if (strpos($k, "[") > 0 && strpos($k, "]") == strlen($k) - 1) {
                // Its a bracketet field, like filename[text] which should be separated and put into a named array.
                list($k, $h) = explode("[", $k, 2);
                if (strlen($k) > 0 && strlen($h) > 1) {
                    $h = rtrim($h, "]");
                    if (!array_key_exists($k, $_POST) || !is_array($_POST[$k])) {
                        $_POST[$k] = [];
                    }
                    $_POST[$k][$h] = $v;
                }
            } else {
                $_POST[$k] = $v;
            }
        }

        // Now create the mform and update the module...
        // Update_moduleinfo() actually needs the mform somehow. Hence the ugly hacks.
        $mform = new $mformclassname($data, $cw->section, $cm, $course);
        $mform->set_data($data);

        if ($fromform = $mform->get_data()) {
            update_moduleinfo($cm, $fromform, $course, $mform);
            return success::success()->model();
        } else {
            return success::fail("MForm did not recognize submission data")->model();
        }
    }

    /************************
     *                      *
     * edit_period       *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function get_period
     */
    public static function get_period_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of study item'),
        ]);
    }

    /**
     * Return value description for webservice function get_period
     */
    public static function get_period_returns(): external_description {
        return period::structure();
    }

    /**
     * Get period information
     * @param mixed $id Id of period to retrieve
     * @return array
     */
    public static function get_period($id) {
        $p = period::find_by_id($id);
        // Public data - no rights check needed.
        external_api::validate_context($p->page()->studyplan()->context());
        return $p->model();
    }

    /**
     * Parameter description for webservice function edit_period
     */
    public static function edit_period_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of study item'),
            "fullname" => new external_value(PARAM_TEXT, 'Full name of period'),
            "shortname" => new external_value(PARAM_TEXT, 'Short name of period'),
            "startdate" => new external_value(PARAM_TEXT, 'start date of period'),
            "enddate" => new external_value(PARAM_TEXT, 'end date of period'),
        ]);
    }

    /**
     * Return value description for webservice function edit_period
     */
    public static function edit_period_returns(): external_description {
        return period::structure();
    }

    /**
     * Edit period information
     * @param mixed $id
     * @param mixed $fullname
     * @param mixed $shortname
     * @param mixed $startdate
     * @param mixed $enddate
     * @return array
     */
    public static function edit_period($id, $fullname, $shortname, $startdate, $enddate) {

        $p = period::find_by_id($id);
        webservicehelper::require_capabilities(self::CAP_EDIT, $p->page()->studyplan()->context());

        $p->edit([
        'fullname' => $fullname,
        'shortname' => $shortname,
        'startdate' => $startdate,
        'enddate' => $enddate,
        ]);
        return $p->model();
    }

    /************************
     *                      *
     * Change course timing *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function course_period_timing
     */
    public static function course_period_timing_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "period_id" => new external_value(PARAM_INT, 'Period number within page'),
            "course_id" => new external_value(PARAM_INT, 'Id of course to adjust dates for'),
            "span" => new external_value(PARAM_INT, 'Period span (default 1)', VALUE_DEFAULT),
        ]);
    }

    /**
     * Return value description for webservice function course_period_timing
     */
    public static function course_period_timing_returns(): external_description {
        return courseinfo::editor_structure();
    }

    /**
     * Match course start/end to period start/end times
     * @param mixed $periodid
     * @param mixed $courseid
     * @param int $span
     * @return array
     */
    public static function course_period_timing($periodid, $courseid, $span = 1) {
        global $DB;
        $period = period::find_by_id($periodid);
        $periodnr = $period->period();
        $page = $period->page();
        // Check for studyplan edit permissions.
        webservicehelper::require_capabilities(self::CAP_EDIT, $page->studyplan()->context());
        $course = \get_course($courseid);
        $coursecontext = context_course::instance($courseid);

        if (webservicehelper::has_capabilities("moodle/course:update", $coursecontext)) {

            // Get the proper list of all the periods for this page.
            $periods = period::find_for_page($page);

            $pstart = $periods[$periodnr];

            // Determine end period number - Clip span between 1 and last period.
            if ($span <= 1) {
                $pend = $pstart;
            } else if ($periodnr + ($span - 1) > $page->periods()) {
                $pend = $periods[$page->periods()];
            } else {
                $pend = $periods[$periodnr + ($span - 1)];
            }

            // First disable the automatic end date option, since it messes with the timing.
            // Unfortunately there is still no default option to turn this off .
            $record = $DB->get_record("course_format_options", ["courseid" => $course->id, "name" => "automaticenddate"]);
            if ($record && $record->value) {
                $record->value = false;
                $DB->update_record("course_format_options", $record);
            }

            // Actually perform the timing changes, while also updating the module times.
            // Like what happens on a course "reset".
            reset_course_userdata((object)[
                'id' => $course->id,
                'reset_start_date' => $pstart->startdate()->getTimestamp(),
                'reset_end_date' => $pend->enddate()->getTimestamp(),
                'reset_start_date_old' => $course->startdate,
                'reset_end_date_old' => $course->enddate,
            ]);
            // Purge course cache so the dates are properly reflected.
            \course_modinfo::purge_course_cache($course->id);

            return (new courseinfo($course->id))->editor_model();
        } else {
            // Probably should return a nice message.
            throw new \webservice_access_exception("You do not have date change permissions on this course");
        }
    }

    /**
     * Parameter description for webservice function set_studyitem_span
     */
    public static function set_studyitem_span_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of study item'),
            "span" => new external_value(PARAM_INT, 'span of item'),
        ]);
    }

    /**
     * Return value description for webservice function set_studyitem_span
     */
    public static function set_studyitem_span_returns(): external_description {
        return studyitem::editor_structure();
    }

    /**
     * Set studyitem span to one or more periods
     * @param mixed $id
     * @param null $span
     * @return array
     */
    public static function set_studyitem_span($id, $span = null) {
        $o = studyitem::find_by_id($id);
        $studyplan = $o->studyline()->studyplan();
        if ($studyplan->is_coach() && !$studyplan->suspended()) {
            external_api::validate_context($studyplan->context());
        } else {
            webservicehelper::require_capabilities(self::CAP_EDIT, $o->context());
        }

        $config = [ 'span' => $span];
        $o->edit($config);
        return $o->editor_model();

    }

    /************************
     *                      *
     * bulk_course_timing   *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function bulk_course_timing
     */
    public static function bulk_course_timing_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "page_id" => new external_value(PARAM_INT, 'id of studyplanpage'),
        ] );
    }

    /**
     * Return value description for webservice function bulk_course_timing
     */
    public static function bulk_course_timing_returns(): external_description {
        return success::structure();
    }

    /**
     * Update course timing for all courses in a studyplan page
     * @param mixed $pageid Id of the studyplan page
     * @return array Succes/fail model
     */
    public static function bulk_course_timing($pageid) {
        $page = studyplanpage::find_by_id($pageid);
        // Validate if the requesting user has the right to edit the plan in it's current context.
        webservicehelper::require_capabilities(self::CAP_EDIT, $page->studyplan()->context());
        preloader::preload_courses($page->studyplan());
        $result = true;
        $message = [];
        $periods = period::find_for_page($page);
        $studylines = studyline::find_page_children($page);
        foreach ($studylines as $line) {
            $studyitems = studyitem::find_studyline_children($line);
            foreach ($studyitems as $item) {
                if ($item->type() == studyitem::COURSE) {
                    if ($item->slot() <= $page->periods()) {
                        $period = $periods[$item->slot()];
                        try {
                            self::course_period_timing($period->id(), $item->courseid(), $item->span());
                        } catch (\webservice_access_exception $x) {
                            $result &= false;
                            $course = preloader::get_course($item->courseid());
                            $message[] = "Course {$course->shortname}: {$x->debuginfo}";
                        } catch (\Exception $x) {
                            $result &= false;
                            $course = preloader::get_course($item->courseid());
                            $xclass = \get_class($x);
                            $message[] = "Course {$course->shortname}: {$xclass} {$x->getMessage()}";
                        }
                    }
                }
            }
        }

        return (new success($result, implode("<p>\n", $message)))->model();
    }

    /************************
     *                      *
     * delete_studyplanpage *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function delete_studyplan
     */
    public static function delete_studyplanpage_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of studyplanpage'),
            "force" => new external_value(PARAM_BOOL, 'force deletion', VALUE_DEFAULT),
        ] );
    }

    /**
     * Return value description for webservice function delete_studyplan
     */
    public static function delete_studyplanpage_returns(): external_description {
        return success::structure();
    }

    /**
     * Delete a studyplan
     * @param mixed $id Id of the studyplan
     * @param bool $force Force deletion, even though studyplan is not empty
     * @return array Succes/fail model
     */
    public static function delete_studyplanpage($id, $force = false) {
        $o = studyplanpage::find_by_id($id);
        $p = $o->studyplan();
        // Validate if the requesting user has the right to edit the plan in it's current context.
        webservicehelper::require_capabilities(self::CAP_EDIT, $p->context());
        return $o->delete(!!$force)->model();
    }

    /************************
     *                      *
     * list_roles *
     *                      *
     ************************/

    /**
     * Parameter description for webservice function list_roles
     */
    public static function list_roles_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "studyplan_id" => new external_value(PARAM_INT, 'id of studyplan'),
        ] );
    }

    /**
     * Return value description for webservice function list_roles
     */
    public static function list_roles_returns(): external_description {
        return new external_multiple_structure(new external_single_structure([
            "id" => new external_value(PARAM_INT, 'id of studyline'),
            "name" => new external_value(PARAM_TEXT, 'shortname of studyline'),
        ]));
    }

    /**
     * List available roles
     * @param mixed $studyplanid id of the studyplan to list the roles for
     * @return array List of roles
     */
    public static function list_roles($studyplanid) {
        webservicehelper::system_context();
        $contextlevels = [CONTEXT_SYSTEM, CONTEXT_COURSECAT];
        $list = [];
        $roles = \get_all_roles();
        foreach ($roles as $role) {
            $name = \role_get_name($role); // Get localized role name.
            $rctxtlevels = \get_role_contextlevels($role->id);
            $intersect = array_intersect($rctxtlevels, $contextlevels);
            if (count($intersect) > 0) {
                $list[] = [
                    'id' => $role->id,
                    'name' => $name,
                ];
            }
        }

        return $list;

    }

    /***************************
     *                         *
     * line_enrol_students     *
     *                         *
     ***************************/

     /**
      * Enrol status structure including user info
      * @param int $required VALUE_OPTIONAL or VALUE_REQUIRED
      */
    public static function student_enrol_status_structure($required = VALUE_REQUIRED) {
        return new external_single_structure([
            'user' => associationservice::user_structure(),
            'enrol' => studyline::enrol_info_structure(),
        ], '', $required);
    }

    /**
     * Create student enrol status model
     * @param int $userid Id of user
     * @param studyline $line Studyline to retrieve enrol status for
     */
    public static function student_enrol_status_model($userid, $line) {
        global $DB;
        $r = $DB->get_record('user', ["id" => $userid]);
        return [
            'user' => associationservice::make_user_model($r),
            'enrol' => $line->enrol_info_model($userid),
        ];
    }

    /**
     * Parameter description for webservice function line_enrol_students
     */
    public static function line_enrol_students_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of specific studyline to enrol into'),
            "users" => new external_multiple_structure(new external_value(PARAM_INT), 'list of user ids'),
        ] );
    }

    /**
     * Return value description for webservice function line_enrol_students
     */
    public static function line_enrol_students_returns(): external_description {
        return new external_multiple_structure(self::student_enrol_status_structure());
    }

    /**
     * Enrol users in a study line
     * @param int $id ID of the studyline to enrol in
     * @param array $users ID's of users to enrol
     * @return array
     */
    public static function line_enrol_students($id, $users) {
        global $USER;
        $userid = $USER->id;

        $o = studyline::find_by_id($id);

        /* NOTE: Perhaps the additional check for the view permission is not needed
                 since there is already a check on roles going on....
        */
        $context = $o->context();
        webservicehelper::require_capabilities(self::CAP_VIEW, $context);

        $list = [];
        // Unenrol capability also acts as overriding manager capability to register/unregister.
        $canunenrol = \has_capability('local/treestudyplan:lineunenrol', $context);
        foreach ($users as $userid) {
            if ($o->can_enrol($userid) || $canunenrol) {
                $o->enrol($userid);
                $list[] = self::student_enrol_status_model($userid, $o);
            }
        }

        // Trigger immediate cohort synchronization for this line only.
        autocohortsync::syncline($o);
        return $list;
    }

    /***************************
     *                         *
     * line_unenrol_students     *
     *                         *
     ***************************/

    /**
     * Parameter description for webservice function line_unenrol_students
     */
    public static function line_unenrol_students_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of specific studyline to enrol into'),
            "users" => new external_multiple_structure(new external_value(PARAM_INT), 'list of user ids'),
        ] );
    }

    /**
     * Return value description for webservice function line_unenrol_students
     */
    public static function line_unenrol_students_returns(): external_description {
        return new external_multiple_structure(self::student_enrol_status_structure());
    }

    /**
     * Unenrol users from a study line
     * @param int $id ID of the studyline to unenrol from
     * @param array $users ID's of users to unenroll
     * @return array
     */
    public static function line_unenrol_students($id, $users) {
        global $USER;
        $userid = $USER->id;

        $o = studyline::find_by_id($id);

        /* NOTE: Perhaps the additional check for the view permission is not needed
                 since there is already a check on roles going on....
        */
        $context = $o->context();
        webservicehelper::require_capabilities('local/treestudyplan:lineunenrol', $context);

        $list = [];
        foreach ($users as $userid) {
            $o->unenrol($userid);
            $list[] = self::student_enrol_status_model($userid, $o);
        }

        // Trigger immediate cohort synchronization for this line only.
        autocohortsync::syncline($o);
        return $list;
    }

    /***************************
     *                         *
     * line_enrol_students     *
     *                         *
     ***************************/

    /**
     * Parameter description for webservice function list_line_enrolled_students
     */
    public static function list_line_enrolled_students_parameters(): external_function_parameters {
        return new external_function_parameters( [
            "id" => new external_value(PARAM_INT, 'id of specific studyline to list for'),
        ] );
    }

    /**
     * Return value description for webservice function list_line_enrolled_students
     */
    public static function list_line_enrolled_students_returns(): external_description {
        return new external_single_structure([
            "userinfo" => new external_multiple_structure(self::student_enrol_status_structure()),
            "can_unenrol" => new external_value(PARAM_BOOL, "True if the requesting user can unenrol students"),
        ]);
    }

    /**
     * List all students enrolled in a specific enrollable study line
     * @param int $id ID of studyline to retrieve
     * @return array
     */
    public static function list_line_enrolled_students($id) {
        $o = studyline::find_by_id($id);
        $context = $o->context();
        webservicehelper::require_capabilities(self::CAP_VIEW, $context);

        $list = [];
        $p = $o->studyplan();
        foreach ($p->find_linked_userids() as $userid) {
            $list[] = self::student_enrol_status_model($userid, $o);
        }

        return [
            "userinfo" => $list,
            "can_unenrol" => \has_capability('local/treestudyplan:lineunenrol', $context),
        ];
    }

    /***************************
     *                         *
     * count_templates         *
     *                         *
     ***************************/

    /**
     * Parameter description for webservice function count_templates
     */
    public static function count_templates_parameters(): external_function_parameters {
        return new external_function_parameters( [
        ] );
    }

    /**
     * Return value description for webservice function count_templates
     */
    public static function count_templates_returns(): external_description {
        return new external_value(PARAM_INT, "True if the requesting user can unenrol students");
    }

    /**
     * Count the number of templates available
     * @return int
     */
    public static function count_templates() {
        external_api::validate_context(context_system::instance());
        require_login(null, false, null);
        return studyplan::count_template();
    }
}
