<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Webservice return value to describe a single event as successful or not
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan;

use core_external\external_api;
use core_external\external_function_parameters;
use core_external\external_description;
use core_external\external_value;
use core_external\external_multiple_structure;
use core_external\external_single_structure;

/**
 * Describes a webservice result a successful or not with optional message
 */
class success {

    /**
     * Success (true) or failed (false)
     * @var bool
     */
    private $success;
    /**
     * Messsage passed with result
     * @var string
     */
    private $msg;

    /**
     * Create new successful result with optional message
     * @param array|object $data Custom data to pass to receiver
     */
    public static function success($data=[]): self {
        return new self(true, 'success', $data);
    }

    /**
     * Create new failed result with optional message
     * @param string $msg Message to add to result
     * @param array|object $data Custom data to pass to receiver
     */
    public static function fail($msg = "", $data=[]): self {
        return new self(false, $msg, $data);
    }

    /**
     * Create new succes result
     * @param bool $success Whether result is succesful or not
     * @param string $msg Message to add to result
     * @param array|object $data Custom data to pass to receiver
     */
    public function __construct($success, $msg, $data=[]) {
        $this->success = ($success) ? true : false;
        $this->msg = $msg;
        $this->data = json_encode($data);
    }

    /**
     * Describe the result for the webservice model
     */
    public static function structure(): external_description {
        return new external_single_structure([
            "success" => new external_value(PARAM_BOOL, 'operation completed succesfully'),
            "msg" => new external_value(PARAM_RAW, 'message'),
            "data" => new external_value(PARAM_RAW, 'message'),
        ]);
    }

    /**
     * Make the webservice result model
     * @return array Webservice value
     */
    public function model() {
        return ["success" => $this->success, "msg" => $this->msg, "data" => $this->data];
    }

    /**
     * Get success status of object
     *
     * @return bool If this success is successful or not
     */
    public function successful() {
        return $this->success;
    }

    /**
     * Get message
     *
     * @return string Message
     */
    public function msg() {
        return $this->msg;
    }

    /**
     * Get data
     *
     * @return string Message
     */
    public function data() {
        return json_decode($this->data);
    }

}
