<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Developer CLI tool. Primes grade generator with random student properties
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_treestudyplan;

use local_treestudyplan\local\gradegenerator;

define('CLI_SCRIPT', true);
require(__DIR__ . '/../../../config.php');
require_once($CFG->libdir . '/clilib.php');

$usage = "Prime the generator with random stats for all relevant students
Usage:
    # php prime_students.php --studyplan=<shortname>  [--file|-f=userfile]
    # php prime_students.php --all|-a [--file|-f=userfile]
    # php prime_students.php --help|-h

Options:
    -h --help           Print this help.
    --file=<value>      Specify file name to store generated user specs in.
";

[$options, $unrecognised] = cli_get_params([
    'help' => false,
    'studyplan' => null,
    'all' => false,
    'file' => "/tmp/generategrades.json",
], [
    'h' => 'help',
    's' => 'studyplan',
    'a' => 'all',
    'f' => 'file',
]);

if ($unrecognised) {
    $unrecognised = implode(PHP_EOL . '  ', $unrecognised);
    cli_error(get_string('cliunknowoption', 'core_admin', $unrecognised));
}

if ($options['help']) {
    cli_writeln($usage);
    exit(2);
}

if (empty($options['studyplan']) && empty($options["all"])) {
    cli_error('Missing mandatory argument studyplan.', 2);
}

if (!empty($options["all"])) {
    $plans = studyplan::find_all();
} else {
    $plans = studyplan::find_by_shortname($options["studyplan"]);
}

$generator = new gradegenerator();
$generator->from_file($options["file"]);

cli_writeln(count($plans) . " studyplans found:");
foreach ($plans as $plan) {
    cli_heading($plan->name());
    $users = $plan->find_linked_users();
    foreach ($users as $u) {
        $generator->addstudent($u->username);
        $generator->add_username_info($u->username, $u->firstname, $u->lastname);
        cli_writeln("  - {$u->firstname} {$u->lastname} / {$u->username}");
    }

    foreach ($plan->pages() as $page) {
        $lines = studyline::find_page_children($page);
        foreach ($lines as $line) {
            cli_writeln("  ** {$line->name()} **");
            $items = studyitem::find_studyline_children($line);
            foreach ($users as $u) {
                $generator->addskill($u->username, $line->shortname());
            }
        }
    }
}

$generator->to_file($options["file"]);
