<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Edit/manage study plans
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once("../../config.php");
require_once($CFG->libdir . '/weblib.php');

use local_treestudyplan\contextinfo;
use local_treestudyplan\studyplan;
use local_treestudyplan\courseservice;
use local_treestudyplan\debug;
use core\context;
use core\context\coursecat as context_coursecat;
use core\context\system as context_system;

$systemcontext = context_system::instance();
$PAGE->set_context($systemcontext);
$PAGE->set_url("/local/treestudyplan/edit-plan.php", []);
require_login();

// Figure out the context (category or system, based on either category or context parameter).
$categoryid = optional_param('categoryid', 0, PARAM_INT); // Category id.
$contextid = optional_param('contextid', 0, PARAM_INT); // Context id.
if ($categoryid > 0) {
    $studyplancontext = context_coursecat::instance($categoryid);
    if (!is_object($studyplancontext)) {
        $studyplancontext = $systemcontext;
    }
} else if ($contextid > 0) {
    $studyplancontext = context::instance_by_id($contextid);
    if (!is_object($studyplancontext)) {
        $studyplancontext = $systemcontext;
    }
    if (in_array($studyplancontext->contextlevel, [CONTEXT_SYSTEM, CONTEXT_COURSECAT])) {
        $categoryid = $studyplancontext->instanceid;
    } else {
        $studyplancontext = $systemcontext;
    }
} else {
    // If no context is selected, find the first available context for this user.
    $availablecontexts = courseservice::list_available_categories("edit");
    $contextid = 1; // Fallback to system context.
    if (count($availablecontexts) > 0) {
        $contextid = $availablecontexts[0]["context_id"];
    }
    // Reload page with selected category.
    $url = new \moodle_url('/local/treestudyplan/edit-plan.php', ["contextid" => $contextid]);
    header('Location: ' . $url->out(false), true, 302);
    exit;
}

$ci = new contextinfo($studyplancontext);
$contextname = $ci->pathstr();

$PAGE->set_pagelayout('base');
$PAGE->set_title(get_string('cfg_plans', 'local_treestudyplan') . " - " . $contextname);
$PAGE->set_heading($contextname);

if ($studyplancontext->id > 1) {
    navigation_node::override_active_url(new moodle_url('/course/index.php', ['categoryid' => $categoryid ]));
    $PAGE->navbar->add(get_string('cfg_plans', 'local_treestudyplan'));

    // Coursecat context.
    $cat = \core_course_category::get($studyplancontext->instanceid, IGNORE_MISSING, true); // We checck visibility later.
} else {
    // System context.
    $cat = \core_course_category::top();
}

if (!is_object($cat) || !$cat->is_uservisible()) {
    throw new \moodle_exception(
        "error:cannotviewcategory",
        "local_treestudyplan",
        "/local/treestudyplan/edit_plan.php",
        $contextname
    );
}

if (!has_capability('local/treestudyplan:editstudyplan', $studyplancontext)) {
    throw new \moodle_exception(
        "error:nostudyplaneditaccess",
        "local_treestudyplan",
        "/local/treestudyplan/edit_plan.php",
        $contextname
    );
}

// Load page specific css.
$PAGE->requires->css(new moodle_url($CFG->wwwroot . '/local/treestudyplan/css/bootstrap-vue/bootstrap-vue.css'));
if ($CFG->debugdeveloper) {
    $PAGE->requires->css(new moodle_url($CFG->wwwroot . '/local/treestudyplan/styles.css'));
}

print $OUTPUT->header();
print $OUTPUT->render_from_template('local_treestudyplan/edit_plan', [
    "contextid" => $studyplancontext->id,
    "categoryid" => $categoryid,
    "contextname" => $contextname,
]);
print $OUTPUT->footer();
