<?php
// This file is part of the Studyplan plugin for Moodle
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Entry point for external people to access a studyplan view with an invitation
 * @package    local_treestudyplan
 * @copyright  2023 P.M. Kuipers
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once("../../config.php");
use core\context\system as context_system;

/* Since this page is externally accessed and validated in a different way, no login is needed.
   In fact, we need to override the behaviour of require_login to force "guest access" here,
   even if the configuration says differently.
   Note that we do not use set_config(...) here, because we do not want to change the settings
   in the database.
*/
$autologin = $CFG->autologinguests; // Store current guest autologin configuration value.
$CFG->autologinguests = 1; // Override guest autologin configuration value.
require_login(null, true); // Trigger require_login to make sure we have guest access.
$CFG->autologinguests = $autologin; // Restore old guest autologin configuration value.
unset($autologin);

$systemcontext = context_system::instance();
$PAGE->set_pagelayout('base');
$PAGE->set_context($systemcontext);

// See if we can get a valid user for this invited.
$invitekey = optional_param('key', '', PARAM_ALPHANUM);     // Module name.
$PAGE->set_url("/local/treestudyplan/invited.php", ['key' => $invitekey]);

$invite = $DB->get_record_select(
    "local_treestudyplan_invit",
    $DB->sql_compare_text("invitekey") . " = " . $DB->sql_compare_text(":invitekey"),
    ['invitekey' => $invitekey]
);

if (!get_config("local_treestudyplan", "enableplansharing")) {
    $PAGE->set_title(get_string('accessdenied', 'admin'));
    $PAGE->set_heading(get_string('accessdenied', 'admin'));

    print $OUTPUT->header();
    print $OUTPUT->render_from_template('local_treestudyplan/error', [
        "title" => get_string('accessdenied', 'admin'),
        "message" => get_string('error:invitationsdisabled', 'local_treestudyplan'),
    ]);
    print $OUTPUT->footer();

    exit;
} else if (empty($invite)) {
    $PAGE->set_title(get_string('invalid_invitekey_title', 'local_treestudyplan'));
    $PAGE->set_heading(get_string('invalid_invitekey_title', 'local_treestudyplan'));

    print $OUTPUT->header();
    print $OUTPUT->render_from_template('local_treestudyplan/error', [
        "message" => get_string('error:invalid_invitekey', 'local_treestudyplan'),
    ]);
    print $OUTPUT->footer();

    exit;
} else {
    // Load page specific css.
    $PAGE->requires->css(new moodle_url($CFG->wwwroot . '/local/treestudyplan/css/bootstrap-vue/bootstrap-vue.css'));
    if ($CFG->debugdeveloper) {
        $PAGE->requires->css(new moodle_url($CFG->wwwroot . '/local/treestudyplan/styles.css'));
    }

    $student = $DB->get_record('user', ['id' => $invite->user_id]);
    $PAGE->set_title(get_string('report_invited', 'local_treestudyplan', "{$student->firstname} {$student->lastname}"));
    $PAGE->set_heading(get_string('report_invited', 'local_treestudyplan', "{$student->firstname} {$student->lastname}"));

    print $OUTPUT->header();
    print $OUTPUT->render_from_template('local_treestudyplan/invited', ['invitekey' => $invitekey]);
    print $OUTPUT->footer();
}
