<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Report table for the certainty export quiz report.
 * @package    quiz_certainty
 * @copyright  2021 Astor Bizard <astor.bizard@univ-grenoble-alpes.fr>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

require_once(__DIR__ . '/../feedback/feedback_table.php');

use mod_quiz\local\reports\attempts_report_table;
use qbehaviour_certaintywithstudentfbdeferred\certaintylevel;
use qbehaviour_certaintywithstudentfbdeferred\answersubcategory;

/**
 * Report table class definition for the certainty export quiz report.
 * @copyright  2021 Astor Bizard <astor.bizard@univ-grenoble-alpes.fr>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class quiz_certainty_table extends quiz_feedback_table {

    /**
     * @var string $tableid Name of the report table. Used for inheritance, it overrides the parent one.
     */
    protected static $tableid = 'mod-quiz-report-certainty-report';

    /**
     * {@inheritDoc}
     * @see quiz_feedback_table::get_data_fields()
     */
    protected static function get_data_fields() {
        $datafields = parent::get_data_fields();
        $datafields['certainty'] = '-certainty';
        return $datafields;
    }

    /**
     * {@inheritDoc}
     * @see flexible_table::other_cols()
     * @param string $column Table column name.
     * @param object $attempt Quiz attempt data (one attempt per row).
     */
    public function other_cols($column, $attempt) {
        $matches = [];
        if (preg_match('/^answerclass(\d+)$/', $column, $matches)) {
            // This column is an answerclass column.
            $slot = $matches[1];
            if ($attempt->usageid == 0) {
                return '-'; // No data available.
            } else {
                return $this->process_answerclass_column($attempt->usageid, $slot);
            }
        }

        return parent::other_cols($column, $attempt);
    }

    /**
     * {@inheritDoc}
     * @see quiz_feedback_table::process_data_column()
     * @param object $stepdata Attempt step data to replace tagged data.
     * @param string $field Attempt step data field or behaviour var.
     */
    public function process_data_column($stepdata, $field) {
        $res = parent::process_data_column($stepdata, $field);
        if ($field === '-certainty' && $res !== '-') {
            // Format certainty to a readable name.
            $certaintylevel = (int) $res;
            $certainty = certaintylevel::get_levels()[$certaintylevel];
            return trim($certainty->percentage . ' (' . certaintylevel::get_levels()[$certaintylevel]->label . ')');
        } else {
            return $res;
        }
    }

    /**
     * Process answerclass data column.
     * @param int $usageid Question usage id for the attempt.
     * @param int $slot Question slot.
     * @return string Answer class label.
     */
    protected function process_answerclass_column($usageid, $slot) {
        global $CFG;
        require_once($CFG->dirroot . '/question/engine/lib.php');
        $quba = question_engine::load_questions_usage_by_activity($usageid);
        $answersubcategory = answersubcategory::subcategorize_answer($quba->get_question_attempt($slot));
        if ($answersubcategory === null) {
            return '-';
        } else {
            $label = $answersubcategory->answerclass->label;
            if ($this->is_downloading()) {
                return $label;
            } else {
                $icon = html_writer::div('', 'chip',
                        [ 'style' => 'background-color:' . $answersubcategory->color, 'role' => 'presentation' ]);
                return $icon . $label;
            }
        }
    }

    /**
     * {@inheritDoc}
     * @see attempts_report_table::requires_extra_data()
     */
    protected function requires_extra_data() {
        return true;
    }
}
