<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * AJAX script returning detailed information about a question and answer category by chosen answer (for multiple choice questions).
 * @package    report_certaintyquizzes
 * @copyright  2025 Astor Bizard, 2024 Loic Delon
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use qbehaviour_certaintywithstudentfbdeferred\answerclass;
use qbehaviour_certaintywithstudentfbdeferred\answersubcategory;
use report_certaintyquizzes\locallib;

define('AJAX_SCRIPT', true);

require_once(__DIR__ . '/../../../config.php');

require_login();

global $CFG, $PAGE;
require_once($CFG->dirroot . '/question/engine/lib.php');
require_once($CFG->libdir . '/graphlib.php');

$courseid = required_param('courseid', PARAM_INT);
$quizid = required_param('quizid', PARAM_INT);
$questionslot = required_param('questionslot', PARAM_INT);
$attempttype = required_param('attempttype', PARAM_RAW);
$PAGE->set_context(context_course::instance($courseid));

$outcome = new stdClass();
$outcome->success = true;
$outcome->response = new stdClass();
$outcome->error = '';
try {

    $answers = [];

    $userattempts = locallib::get_specific_attempt($quizid, $attempttype);
    $oneqa = question_engine::load_questions_usage_by_activity(reset($userattempts)->uniqueid)->get_question_attempt($questionslot);
    $question = $oneqa->get_question();

    if ($question->get_type_name() === 'truefalse') {
        $qtype = 'truefalse';
        // Process answers.
        foreach ([ 'true', 'false' ] as $field) {
            $answers[$question->{$field . 'answerid'}] = [
                    'iscorrect' => ($question->rightanswer ? 'true' : 'false') === $field,
                    'text' => get_string($field, 'qtype_truefalse'),
            ];
        }
    } else if ($question->get_type_name() === 'multichoice' && $question instanceof qtype_multichoice_single_question) {
        $qtype = 'multichoicesingle';
        // Process answers.
        foreach ($question->answers as $answerid => $answer) {
            $answers[$answerid] = [
                    'iscorrect' => question_state::graded_state_for_fraction($answer->fraction)->is_correct(),
                    'text' => trim($question->html_to_text($answer->answer, $answer->answerformat)),
            ];
        }
    } else {
        // Not supported question type.
        $qtype = null;
    }

    if ($qtype !== null) {
        foreach ($answers as &$answerref) {
            $answerref['label'] = $answerref['text'] . ($answerref['iscorrect'] ? ' (correct)' : ' (incorrect)');
            foreach (answerclass::get_classes() as $answerclass) {
                $answerref['n' . $answerclass->name] = 0;
            }
        }

        foreach ($userattempts as $userattempt) {
            $qa = question_engine::load_questions_usage_by_activity($userattempt->uniqueid)->get_question_attempt($questionslot);

            if ($qtype === 'truefalse') {
                $useranswerid = $question->{($qa->get_last_qt_var('answer') ? 'true' : 'false') . 'answerid'};
            } else if ($qtype === 'multichoicesingle') {
                $useranswerid = explode(',', $qa->get_last_qt_var('_order'))[$qa->get_last_qt_var('answer')];
            } else {
                // Not supported question type (should not happen here).
                continue;
            }

            $subcategory = answersubcategory::subcategorize_answer($qa);
            if ($subcategory !== null) {
                $answers[$useranswerid]['n' . $subcategory->answerclass->name] += 1;
            }

        }

        ksort($answers);

        $answerclasseschart = new \core\chart_bar();
        $answerclasseschart->set_labels(array_column($answers, 'label'));
        foreach (answerclass::get_classes() as $answerclass) {
            $series = new \core\chart_series(get_string($answerclass->name . 'plural', locallib::BEHAVIOURCOMPONENT),
                    array_column($answers, 'n' . $answerclass->name));
            $series->set_color($answerclass->color);
            $answerclasseschart->add_series($series);
        }
        $answerclasseschart->set_stacked(true);
        $answerclasseschart->set_title(get_string('answercategoriesbyanswer', locallib::COMPONENT));

        $outcome->response->chartdata = json_encode($answerclasseschart);

        $outcome->response->questionsummary = html_writer::tag('h6', get_string('labelvalue', 'moodle',
                [ 'label' => get_string('question'), 'value' => format_string($question->name) ]));
        $outcome->response->questionsummary .= $question->format_questiontext($oneqa);
        if (!empty($answers)) {
            $answerslist = '<ul>';
            foreach (array_values($answers) as $i => $answer) {
                $str = 'choicetitle' . ($answer['iscorrect'] ? 'correct' : '');
                $line = get_string($str, locallib::COMPONENT, [ 'i' => $i + 1, 'text' => $answer['text'] ]);
                $answerslist .= html_writer::tag('li', $line, $answer['iscorrect'] ? [ 'class' => 'text-success' ] : []);
            }
            $answerslist .= '</ul>';
            $outcome->response->questionsummary .= $answerslist;
        }
    }

} catch (Exception $e) {
    $outcome->success = false;
    $outcome->error = $e->getMessage();
}
echo json_encode($outcome);
die();
