<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * AJAX script returning attempt data details for given quiz, users and attempt type.
 * @package    report_certaintyquizzes
 * @copyright  2025 Astor Bizard, 2024 Loic Delon
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use report_certaintyquizzes\locallib;

define('AJAX_SCRIPT', true);

require_once(__DIR__ . '/../../../config.php');

require_login();

global $PAGE;

$courseid = required_param('courseid', PARAM_INT);
$usersids = required_param_array('usersids', PARAM_INT);
$quizid = required_param('quizid', PARAM_INT);
$attempttype = required_param('attempttype', PARAM_RAW);
$sortcolumn = optional_param('sortcolumn', '', PARAM_RAW);
$PAGE->set_context(context_course::instance($courseid));

$outcome = new stdClass();
$outcome->success = true;
$outcome->response = null;
$outcome->error = '';
try {
    $hasdeclaredignorance = \qbehaviour_certaintywithstudentfbdeferred\locallib::exists_level_of_declared_ignorance();
    $table = new html_table();
    $table->head = [
            get_string('defaultcoursestudent'),
            get_string('attemptnumber', 'mod_quiz'),
            get_string('gradeover100', locallib::COMPONENT),
            get_string('lucidityindex', locallib::BEHAVIOURCOMPONENT),
            get_string('metacognitiveindicators', locallib::BEHAVIOURCOMPONENT),
    ];
    $table->headspan = [
            1,
            1,
            1,
            1,
            $hasdeclaredignorance ? 2 : 1,
    ];
    $table->attributes['class'] = 'student-attempts-table';
    $table->responsive = false;
    foreach (locallib::get_specific_attempt($quizid, $attempttype, $usersids) as $userid => $attempt) {

        $userfullname = fullname(\core_user::get_user($userid, implode(',', \core_user\fields::get_name_fields())));
        $attemptnum = get_string('attemptxofy', locallib::COMPONENT, [ 'x' => $attempt->attempt, 'y' => $attempt->totalattempts ]);
        $attemptdata = \qbehaviour_certaintywithstudentfbdeferred\locallib::get_lucidity_indicators_for_attempt($attempt->uniqueid,
                \qbehaviour_certaintywithstudentfbdeferred\locallib::INDICATOR_GRADE |
                \qbehaviour_certaintywithstudentfbdeferred\locallib::INDICATOR_LUCIDITY |
                \qbehaviour_certaintywithstudentfbdeferred\locallib::INDICATOR_LUCIDITY_LABEL_COLOR |
                \qbehaviour_certaintywithstudentfbdeferred\locallib::INDICATOR_IGNORANCERATE
                );

        $row = [
                html_writer::link(new moodle_url('/user/profile.php', [ 'id' => $userid, 'course' => $courseid ]), $userfullname),
                html_writer::link(new moodle_url('/mod/quiz/review.php', [ 'attempt' => $attempt->id ]), $attemptnum),
                $attemptdata->gradeover100,
                $attemptdata->lucidityindex,
        ];

        if ($hasdeclaredignorance) {
            $dircircle = html_writer::span('', 'circle-indicator mr-2',
                    [ 'style' => '--shape-color:' . $attemptdata->ignorancerateindicatorcolor ]);
            $ignoranceratelabel = html_writer::span(get_string('dira', locallib::COMPONENT, $attemptdata->declaredignorancerate));
            $row[] = html_writer::div($dircircle . $ignoranceratelabel, 'd-flex align-items-center text-nowrap text-left');
        }

        $answerclasscircle = html_writer::span('', 'circle-indicator mr-2',
                [ 'style' => '--shape-color:' . $attemptdata->lucidityindicatorcolor ]);
        $row[] = html_writer::div($answerclasscircle . $attemptdata->lucidityindicatorlabel, 'd-flex align-items-center text-left');

        $table->data[] = $row;
    }

    if (in_array($sortcolumn, [ 'grade', 'lucidityindex' ])) {
        $i = $sortcolumn === 'grade' ? 2 : 3;
        usort($table->data, function($row1, $row2) use ($i) {
            return $row1[$i] - $row2[$i];
        });
    }

    $outcome->response = html_writer::table($table);

} catch (Exception $e) {
    $outcome->success = false;
    $outcome->error = $e->getMessage();
}
echo json_encode($outcome);
die();
