// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * JS for this plugin.
 *
 * @copyright   IntegrityAdvocate.com
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

class block_quizonepagepaginate {
    constructor(versionstring, questionsperpage) {
        const debug = false;
        const self = this;
        const fxn = self.constructor.name + '.constructor';
        window.console.log(fxn + '::Started with versionstring=' + versionstring + '; questionsperpage=' + questionsperpage);

        if (!self.isAQuizAttemptPage()) {
            debug && window.console.log(fxn + '::We should not use this block JS bc this is not a quiz attempt page');
            return;
        }

        if (isNaN(questionsperpage)) {
            throw new Error(fxn + '::Invalid value passed for param questionsperpage');
        }

        // How many quiz questions to show at one time.
        self.questionsperpage = parseInt(questionsperpage);
        if (self.questionsperpage < 1) {
            throw new Error(fxn + '::Invalid value passed for param questionsperpage');
        }

        // The index of the first quiz question to show.
        self.firstQuestionToShow = 0;

        // Used to locate the quiz questions on the page.
        self.eltQuestionsSelector = '#page-mod-quiz-attempt #responseform .que';
        // Used to place this plugin's JS-driven next/prev nav buttons.
        self.eltQuizFinishAttemptButtonSelector = '#responseform .submitbtns .mod_quiz-next-nav';
        // Button to show tne previous questions.
        self.eltBqoppButtonPrev = self.constructor.name + '-prev';
        // Button to show tne next questions.
        self.eltBqoppButtonNext = self.constructor.name + '-next';

        // Holds all the current page quiz questions, visible or not.
        self.arrQuestions = [];
    }

    run() {
        const debug = false;
        const self = this;
        const fxn = self.constructor.name + '.run';
        debug && window.console.log(fxn + '::Started with self.firstQuestionToShow=; self.questionsperpage=', self.firstQuestionToShow, self.questionsperpage);

        if (!self.isAQuizAttemptPage() || !self.shouldQuizPaginate()) {
            window.console.log(fxn + '::We should not use this block JS: self.isAQuizAttemptPage()=' + self.isAQuizAttemptPage() + '; self.shouldQuizPaginate()=' + self.shouldQuizPaginate());
            return;
        }

        self.getAllQuestions();

        debug && window.console.log(fxn + '::About to self.addNextPrevButtons()');
        self.addNextPrevButtons();

        // Handle changes to URL anchor.
        window.addEventListener('hashchange', self.handleAnchorChange);

        // Find the question index matching the question-* number.
        const requestedQuestionIndex = self.getAnchorQuestionIndex(document.URL);
        debug && window.console.log(fxn + '::Got requestedQuestionIndex=', requestedQuestionIndex);
        if (requestedQuestionIndex >= 0) {
            self.firstQuestionToShow = requestedQuestionIndex;
        }
        self.hideShowQuestions(self.firstQuestionToShow, self.questionsperpage);
    }

    isAQuizAttemptPage() {
        return document.body.id === 'page-mod-quiz-attempt';
    }

    shouldQuizPaginate() {
        const debug = false;
        const self = this;
        const fxn = self.constructor.name + '.shouldQuizPaginate';
        debug && window.console.log(fxn + '::Started');

        let shouldQuizPaginate = false;

        // Populate self.arrQuestions if not already done.
        self.getAllQuestions();

        shouldQuizPaginate = self.questionsperpage < self.arrQuestions.length;
        debug && window.console.log(fxn + '::Got self.questionsperpage=' + self.questionsperpage + ' is it < self.arrQuestions.length=' + self.arrQuestions.length, shouldQuizPaginate);

        return shouldQuizPaginate;
    }

    /**
     * If the URL anchor value matches /question-\d+-\d+/, get the index of the self.arrQuestions item that matches.
     *
     * @param {string} url URL containing the anchor e.g. "https://my.moodle.com/mod/quiz/attempt.php?attempt=58&cmid=3#question-23-9".
     * @returns {number} The matching index in self.arrQuestions; else -1.
     */
    getAnchorQuestionIndex(url = '') {
        const debug = false;
        const self = this;
        const fxn = self.constructor.name + '.getAnchorQuestionIndex';
        debug && window.console.log(fxn + '::Started');

        let questionIndex = -1;

        const anchor = self.getAnchor(url);
        debug && window.console.log(fxn + '::Got anchor=', anchor);
        if (!anchor || anchor.length < 'question-1-1'.length) {
            debug && window.console.log(fxn + '::No anchor  so return questionIndex=', questionIndex);
            return questionIndex;
        }

        const questionNrRequested = self.getAnchorQuestionNr(anchor);
        if (!questionNrRequested) {
            return questionIndex;
        }

        questionIndex = self.findQuestionIndexFromQuestionNr(questionNrRequested);

        return questionIndex;
    }

    /**
     * Get the URL anchor value.
     *
     * @param {string} url A URL to get the anchor value from e.g. "https://my.moodle.com/mod/quiz/attempt.php?attempt=58&cmid=3#blah".
     * @returns {string} The URL anchor value (e.g. "blah" in url=https://my.moodle.com/mod/quiz/attempt.php?attempt=58&cmid=3#blah); else return empty string.
     */
    getAnchor(url = '') {
        if (!url || url.length < 1 || typeof url !== 'string') {
            return '';
        }

        const anchor = url.split("#")[1];
        return anchor ? anchor : "";
    }

    /**
     * Extract the question sequence number from the URL anchor text.
     *
     * @param {string} anchor The URL anchor string (e.g. "blah" in url=https://my.moodle.com/mod/quiz/attempt.php?attempt=58&cmid=3#blah).
     * @returns {string} The question number e.g. "question-23-9" from the URL anchor value (e.g. from https://my.moodle.com/mod/quiz/attempt.php?attempt=58&cmid=3#question-23-9); else return empty string.
     */
    getAnchorQuestionNr(anchor = '') {
        const debug = false;
        const self = this;
        const fxn = self.constructor.name + '.getAnchorQuestionNr';
        debug && window.console.log(fxn + '::Started');

        // This value is in the format mm-nn where mm=the quiz attempt number; nn=the question index.
        let questionNrRequested = '';

        if (anchor && anchor.length > 2) {
            const regexResults = anchor.match(/(question-\d+-\d+)/);
            debug && window.console.log(fxn + '::Got regexResults=', regexResults);
            if (regexResults) {
                questionNrRequested = regexResults[1];
            }
        }
        debug && window.console.log(fxn + '::Got questionNrRequested=', questionNrRequested);

        return questionNrRequested;
    }

    /**
     * Search self.arrQuestions for a question with number=questionNr.
     *
     * @param {str} questionNr The question number e.g. "question-23-9".
     * @returns {number} The index of self.arrQuestions that matches; else -1.
     */
    findQuestionIndexFromQuestionNr(questionNr = '') {
        const debug = false;
        const self = this;
        const fxn = self.constructor.name + '.findQuestionIndexFromQuestionNr';
        debug && window.console.log(fxn + '::Started with questionNr=', questionNr);

        let indexFound = -1;

        if (!questionNr || typeof questionNr !== 'string' || questionNr.length < 'question-1-1'.length) {
            window.console.log(fxn + '::Invalid value passed for questionNr so return not found');
            return indexFound;
        }
        if (self.arrQuestions.length < 1) {
            window.console.log(fxn + '::arrQuestions is empty so return not found');
            return indexFound;
        }

        self.arrQuestions.forEach((elt, index) => {
            debug && window.console.log(fxn + '::Looking at index=' + index + '; elt=', elt);
            if (elt.id === questionNr) {
                debug && window.console.log(fxn + '.forEach::Found matching index=', index);
                indexFound = index;
                return;
            }
        });

        debug && window.console.log(fxn + '::About to return indexFound=', indexFound);
        return indexFound;
    }

    getAllQuestions() {
        const debug = false;
        const self = this;
        const fxn = self.constructor.name + '.getAllQuestions';
        debug && window.console.log(fxn + '::Started');

        // If self.arrQuestions is already populated, just return it.
        if (typeof self.arrQuestions != 'undefined' && self.arrQuestions.length > 0) {
            debug && window.console.log(fxn + '::self.arrQuestions is already populated so just return it');
            return self.arrQuestions;
        }

        self.arrQuestions = document.querySelectorAll(self.eltQuestionsSelector);
        debug && window.console.log(fxn + '::Found ' + self.arrQuestions.length + ' questions on the page');

        return self.arrQuestions;
    }

    hideShowQuestions(first = 0, length) {
        const debug = false;
        const self = M.block_quizonepagepaginate;
        const fxn = self.constructor.name + '.hideShowQuestions';
        debug && window.console.log(fxn + '::Started with first=' + first + '; length=' + length);

        if (isNaN(first) || isNaN(length) || first < 0 || length < 1) {
            throw new Error(fxn + '::Invalid value passed for param first or length');
        }
        if (self.arrQuestions.length < 1) {
            throw new Error(fxn + '::self.arrQuestions is empty');
        }

        const last = first + length;
        let countVisible = 0;

        self.arrQuestions.forEach((elt, index) => {
            debug && window.console.log(fxn + '::Looking at question index=' + index + '; elt=', elt);
            if (index >= first && index < last && countVisible < self.questionsperpage) {
                debug && window.console.log(fxn + '::Show this elt  ', elt);
                elt.classList.remove('quizonepage-hidden');
                countVisible++;
            } else {
                debug && window.console.log(fxn + '::Hide this elt', elt);
                elt.classList.add('quizonepage-hidden');
            }
        });

        // Update button visibility after showing/hiding questions.
        self.updatePrevNextButtonVisibility();
    }

    addNextPrevButtons() {
        const debug = false;
        const self = this;
        const fxn = self.constructor.name + '.addNextPrevButtons';
        debug && window.console.log(fxn + '::Started with self.eltQuizFinishAttemptButtonSelector=', self.eltQuizFinishAttemptButtonSelector);

        const eltCloneSource = document.querySelector(self.eltQuizFinishAttemptButtonSelector);
        if (eltCloneSource === null) {
            throw new Error(fxn + '::No button found to clone');
        }

        // String are returned in a plain array in the same order specified here.
        // E.g. [0 => "Previous", 1 => "Next"].
        const stringsToRetrieve = [{
            key: 'previous',
            component: 'core'
        },
        {
            key: 'next',
            component: 'core',
        }
        ];

        // We need core/str bc we use some of the strings for the UI.
        require(['core/str'], function(str) {
            debug && window.console.log(fxn + '.require::Started with stringsToRetrieve=', stringsToRetrieve);

            str.get_strings(stringsToRetrieve).then(
                function(stringsRetrieved) {
                    debug && window.console.log(fxn + '.require.get_strings.then::Started with stringsRetrieved=', stringsRetrieved);

                    const eltPrevInDom = self.addPrevNextButton(eltCloneSource, 'prev', stringsRetrieved);
                    eltPrevInDom.addEventListener('click', self.buttonClickedPrev);

                    const eltNextInDom = self.addPrevNextButton(eltCloneSource, 'next', stringsRetrieved);
                    eltNextInDom.addEventListener('click', self.buttonClickedNext);

                    return stringsRetrieved;
                }).catch(function(err) {
                    console.error(fxn + '::Failed to get strings', err);
                    throw err; // Re-throw to propagate error
                });
        });
    }

    /**
     * Add buttons to the page to JS-navigate through the quiz questions on the page.
     *
     * @param {DomElement} eltCloneSource An existing button in the form buttons area.
     * @param {string} nextorprev Which button to create; valid values=[prev, next]
     * @param {Array<string>} strings Moodle lang strings for the buttons in the order they are created.
     * @returns {DomElement} The DomElement we just inserted.
     */
    addPrevNextButton(eltCloneSource, nextorprev, strings) {
        const debug = false;
        const self = M.block_quizonepagepaginate;
        const fxn = self.constructor.name + '.addPrevNextButton';
        debug && window.console.log(fxn + '::Started wih eltCloneSource=; nextorprev=; strings=', eltCloneSource, nextorprev, strings);

        // Validate params.
        if (!(eltCloneSource instanceof Element)) {
            throw new Error(fxn + '::Invalid value passed for param eltCloneSource');
        }
        if (nextorprev !== 'prev' && nextorprev !== 'next') {
            throw new Error(fxn + '::Invalid value passed for param nextorprev');
        }
        if (!Array.isArray(strings) || strings.length < 2) {
            throw new Error(fxn + '::Invalid value passed for param strings');
        }
        debug && window.console.log(fxn + '::Done validating params');

        const isPrev = nextorprev === 'prev';
        const btnname = (isPrev ? self.eltBqoppButtonPrev : self.eltBqoppButtonNext);
        const btnvalue = strings[(isPrev ? 0 : 1)];

        // The param=true keeps attributes but not listeners.
        const eltClone = eltCloneSource.cloneNode(true);
        eltClone.setAttribute('id', btnname);
        eltClone.className = eltClone.className.replace('btn-primary', 'btn-secondary');
        eltClone.name = btnname;
        eltClone.type = 'button'; // Prevents MacOS from navigating when type=submit.
        eltClone.setAttribute('value', btnvalue); // Safari fix.
        eltClone.setAttribute('data-initial-value', btnvalue);
        eltClone.removeAttribute('disabled');

        // Update button visibility after adding.
        self.updatePrevNextButtonVisibility();

        return eltCloneSource.parentNode.insertBefore(eltClone, eltCloneSource);
    }

    updatePrevNextButtonVisibility() {
        const debug = false;
        const self = M.block_quizonepagepaginate;
        const fxn = self.constructor.name + '.updatePrevNextButtonVisibility';
        debug && window.console.log(fxn + '::Started');

        const prevBtn = document.getElementById(self.eltBqoppButtonPrev);
        const nextBtn = document.getElementById(self.eltBqoppButtonNext);

        if (prevBtn) {
            if (self.firstQuestionToShow <= 0) {
                prevBtn.setAttribute('disabled', 'disabled');
                prevBtn.classList.add('disabled');
            } else {
                prevBtn.removeAttribute('disabled');
                prevBtn.classList.remove('disabled');
            }
        }
        if (nextBtn) {
            const lastPageStart = self.arrQuestions.length - self.questionsperpage;
            if (self.firstQuestionToShow >= lastPageStart) {
                nextBtn.setAttribute('disabled', 'disabled');
                nextBtn.classList.add('disabled');
            } else {
                nextBtn.removeAttribute('disabled');
                nextBtn.classList.remove('disabled');
            }
        }
    }

    buttonClickedPrev() {
        const debug = false;
        const self = M.block_quizonepagepaginate;
        const fxn = self.constructor.name + '.buttonClickedPrev';
        debug && window.console.log(fxn + '::Started');

        self.triggerAutosave();
        self.updateVisibleQuestionRange(false);
        self.hideShowQuestions(self.firstQuestionToShow, self.questionsperpage);
        self.updatePrevNextButtonVisibility();
    }

    buttonClickedNext() {
        const debug = false;
        const self = M.block_quizonepagepaginate;
        const fxn = self.constructor.name + '.buttonClickedNext';
        debug && window.console.log(fxn + '::Started');

        self.triggerAutosave();
        self.updateVisibleQuestionRange(true);
        self.hideShowQuestions(self.firstQuestionToShow, self.questionsperpage);
        self.updatePrevNextButtonVisibility();
    }

    triggerAutosave() {
        const debug = false;
        const self = M.block_quizonepagepaginate;
        const fxn = self.constructor.name + '.triggerAutosave';
        debug && window.console.log(fxn + '::Started');

        try {
            debug && window.console.log(fxn + '::About to trigger autosave');
            M.mod_quiz.autosave.save_changes();
        } catch (error) {
            window.console.log(fxn + '::autosave is disabled');
        }
    }

    updateVisibleQuestionRange(getNextSet = true) {
        const debug = false;
        const self = M.block_quizonepagepaginate;
        const fxn = self.constructor.name + '.updateVisibleQuestionRange';
        debug && window.console.log(fxn + '::Started with getNextSet=', getNextSet);

        const firstOfAllQs = 0;
        const lengthToShow = self.questionsperpage;
        const lastOfAllQs = self.arrQuestions.length;
        debug && window.console.log(fxn + '::Start; firstOfAllQs=' + firstOfAllQs + '; lengthToShow=' + lengthToShow + '; lastOfAllQs=' + lastOfAllQs);

        if (getNextSet) {
            // Propose to jump to the next set of questions.
            const proposedStart = self.firstQuestionToShow + lengthToShow;
            debug && window.console.log(fxn + '::Proposed start of the next set of questions=', proposedStart);

            // Check that the [proposed range of setLength questions] is within the [total range of questions].
            if (proposedStart + lengthToShow < lastOfAllQs) {
                self.firstQuestionToShow = proposedStart;
                debug && window.console.log(fxn + '::The proposedStart + lengthToShow is below the max range, so set self.firstQuestionToShow=', self.firstQuestionToShow);
            } else {
                self.firstQuestionToShow = lastOfAllQs - lengthToShow;
                debug && window.console.log(fxn + '::The proposedStart + lengthToShow is above the max range, so set self.firstQuestionToShow=', self.firstQuestionToShow);
            }
        } else {
            // Propose to jump to the next set of questions.
            const proposedStart = self.firstQuestionToShow - lengthToShow;
            window.console.log(fxn + '::Proposed start of the next set of questions=', proposedStart);

            // Check that the [proposed range of setLength questions] is within the [total range of questions].
            if (proposedStart < firstOfAllQs) {
                debug && window.console.log(fxn + '::The proposedStart is below the min range, so set self.firstQuestionToShow=', self.firstQuestionToShow);
                self.firstQuestionToShow = firstOfAllQs;
            } else {
                debug && window.console.log(fxn + '::The proposedStart is within the min range, so set self.firstQuestionToShow=', self.firstQuestionToShow);
                self.firstQuestionToShow = proposedStart;
            }
        }

        debug && window.console.log(fxn + '::Done; firstOfAllQs=' + firstOfAllQs + '; lengthToShow=' + lengthToShow + '; lastOfAllQs=' + lastOfAllQs);
    }

    /**
     * If the user clicks a quiz navigation block link or types in a URL anchor, handle it here.
     * @param {*} e Event object from the event listener.
     * @returns void
     */
    handleAnchorChange(e) {
        const debug = false;
        const self = M.block_quizonepagepaginate;
        const fxn = self.constructor.name + '.handleAnchorChange';
        debug && window.console.log(fxn + '::Started with e=', e);

        const target = e.target || e.srcElement;
        debug && window.console.log('Found target=', target);

        // Only continue if are working from a valid source.
        let foundHref = '';

        // Handle typed-in URL anchor changes.
        if (self.isWindowObj(target)) {
            foundHref = window.location.href;
            debug && window.console.log(fxn + '::Found window href=', foundHref);
        }

        // Handle mod_quiz_navblock anchor clicks.
        if (foundHref.length < 1) {
            // Is target a child of a mod_quiz_navblock instance?
            const eltBlock = target.closest('#mod_quiz_navblock');
            if (!eltBlock) {
                debug && window.console.log('The target is not a child of the quiz navigation block so just return');
                return;
            }

            // In mod_quiz_navblock the target is a span that is a child of the a element, so get the a element and check it is a Quiz Navigation button.
            const closestA = target.closest('a.qnbutton');
            debug && window.console.log('Found closestA=', closestA);
            if (!closestA) {
                debug && window.console.log('This is not a targeted element so just return');
                return;
            }

            foundHref = closestA.href;
            debug && window.console.log('Found foundHref', foundHref);
        }

        if (foundHref.length < 1) {
            debug && window.console.log('No valid href found so just return');
            return;
        }

        const requestedQuestionIndex = self.getAnchorQuestionIndex(foundHref);
        debug && window.console.log(fxn + '::Got requestedQuestionIndex=', requestedQuestionIndex);
        if (requestedQuestionIndex >= 0) {
            self.firstQuestionToShow = requestedQuestionIndex;
        }
        self.hideShowQuestions(self.firstQuestionToShow, self.questionsperpage);
    }

    isWindowObj(obj) {
        return obj && obj.document && obj.location && obj.alert && obj.setInterval;
    }
}

/**
 * Setup the module.
 *
 * @param {string} versionstring The Moodle version string e.g. "2022090900".
 * @param {number} questionsperpage How many quiz questions to show at once.
 */
export const init = (versionstring, questionsperpage) => {
    const debug = false;
    const fxn = 'block_quizonepagepaginate::init';
    debug && window.console.log(fxn + '::Started with versionstring=' + versionstring + '; questionsperpage=' + questionsperpage);

    try {
        M.block_quizonepagepaginate = new block_quizonepagepaginate(versionstring, questionsperpage);
        // Disabled bc not needed: debug && window.console.log('M.block_quizonepagepaginate::Built class=', M.block_quizonepagepaginate);
        M.block_quizonepagepaginate.run();
    } catch (e) {
        window.console.error(e);
    }
};
